/*******************************************************************
 pic-logger for PIC18F14K50					Create Date	2011/05/05
 FileName:      main.c						Last UpDate	2013/01/15
 Complier:  	Microchip C18						Saka Softwares

 TAB size = 4
/** INCLUDES *******************************************************/
#include "GenericTypeDefs.h"
#include "Compiler.h"
#include "USB/usb.h"
#include "USB/usb_function_cdc.h"
#include "USB/usb_device.h"

#include "usb_config.h"						/* USB configuretion   */
#include "HardwareProfile.h"				/* pic-logger hardwear */
#include "eeprom.h"							/* EEPROM read/write   */

#define HARDWARE_NAME		"pic-logger\r"	// Hardware Name
#define	HARDWARE_VERSION	"F7\r"			// Hardware Version
#define ADC_BITS			10				// ADC 10-bits binary
/** File Version ***************************************************/
/**	F7		2013/01/15	d荞݂Ƃ̖C
/**	F6		2012/12/31	2KB HID BootloaderΉ \[XJ
/** F5		2012/10/12	\[XC(sv폜yуRgǉ)
/**	F3		2011/06/12	4KB HID BootloaderΉ
/**	F2		2011/06/06	J
/**	F0		2011/05/05	(pic-loggero[Wp)

/** CONFIGURATION **************************************************/
	//PIC18F14K50
	#pragma config CPUDIV = NOCLKDIV
	#pragma config USBDIV = OFF
	#pragma config FOSC   = HS
	#pragma config PLLEN  = ON
	#pragma config FCMEN  = OFF
	#pragma config IESO   = OFF
	#pragma config PWRTEN = ON
	#pragma config BOREN  = ON
	#pragma config BORV   = 30
	#pragma config WDTEN  = OFF
	#pragma config WDTPS  = 32768
	#pragma config MCLRE  = OFF
	#pragma config HFOFST = OFF
	#pragma config STVREN = ON
	#pragma config LVP    = OFF
	#pragma config XINST  = OFF
	#pragma config BBSIZ  = OFF
	#pragma config CP0    = OFF
	#pragma config CP1    = OFF
	#pragma config CPB    = OFF
	#pragma config WRT0   = OFF
	#pragma config WRT1   = OFF
	#pragma config WRTB   = OFF
	#pragma config WRTC   = OFF
	#pragma config EBTR0  = OFF
	#pragma config EBTR1  = OFF
	#pragma config EBTRB  = OFF       

/** VARIABLES ******************************************************/
#pragma udata
#define	USBMAXSND	32					// USB ő呗Mobt@
#define	USBMAXRCV	32					// USB őMobt@
#define ADCFORMAT	5					// ADC ʐMp16bittH[}bg(5oCg)
#define	ADCMAXCHS	6					// ADC őAiO`l
#define	ADCOVRSPL	256					// ADC I[o[TvO (14bit)
#define	ADCSAMPLE	32					// ADC 1̃Tv
#define	ADCBUFFER	8					// ADC f[^obt@
#define	ADCSHFBITS	4					// ADC Vtgrbg
#define	CHANGEWAIT	20					// ADC ؂ւEGCg
#define EPGNDBASE	0					// EEPROM GNDf[^p Base Address
#define EPREFBASE	32					// EEPROM t@Xf[^p Base Address
#define EPVLTBASE	64					// EEPROM t@Xdp Base Address

char UsbSend[USBMAXSND];				// USB Mpobt@
char UsbReceive[USBMAXRCV];				// USB Mpobt@

WORD AdcDat[ADCMAXCHS][ADCBUFFER];		// ADC `l̃f[^
WORD AdcAdd;							// ADC f[^Zp
volatile BOOL bAdcErrorFlag;			// ADC ϊG[tO
BYTE AdcChannel[ADCMAXCHS];				// ADC ANx̓o^p
BYTE AdcRegCh;							// ADC o^ꂽANx̐
BYTE AdcChNum;							// ADC ϊANx
BYTE BufCount;							// obt@pJEg
BYTE SamplCount;						// Tv JEg
BYTE ChageWait;							// `l؂芷̃EFCgJEg
BYTE Switch2;							// XCb`2ԏ

/** PRIVATE  PROTOTYPES ********************************************/
void Initialize(void);
void InterruptsEnable(void);
void InterruptsDisable(void);
void ProcessIO(void);
void USBDeviceTasks(void);
void HighPriorityISRCode(void);
void LowPriorityISRCode(void);
void USBCBSendResume(void);
void BlinkUSBStatus(void);

enum { VCH0, VCH1, VCH2, VCH3, VCH4, VCH5 };
enum { RENG1REF, RENG2REF, RENG4REF };

void ResetToBootloader(void);
void Timer2_Setup(BOOL on);
void Eccp1_Setup(BOOL on);
void ADC_Setup(BOOL on);
void AllReentryAnalogChannel(void);
void EntryAnalogChannel(BYTE channel);
void Vref_Setup(BOOL on);
char* ByteToHex(BYTE dat, char *p);
char* AdcToHex(BYTE adcnum, WORD dat, char *p);
void Hardware_Name(void);
void Hardware_Version(void);
void ADC_Bits(void);
void Number_Samples(void);
WORD AdcHighReso(WORD *p);
void Send_AnalogData(void);
void Read_Ground(void);
void Write_Ground(void);
void Read_Reference(void);
void Write_Reference(void);
void Read_VoltData(void);
void Write_VoltData(void);
void Channel_Number(void);

/** VECTOR REMAPPING ***********************************************/
#if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)
	#define REMAPPED_RESET_VECTOR_ADDRESS			0x1000
	#define REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS	0x1008
	#define REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS	0x1018
#elif defined(PROGRAMMABLE_WITH_USB_MCHPUSB_BOOTLOADER)	
	#define REMAPPED_RESET_VECTOR_ADDRESS			0x800
	#define REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS	0x808
	#define REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS	0x818
#else	
	#define REMAPPED_RESET_VECTOR_ADDRESS			0x00
	#define REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS	0x08
	#define REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS	0x18
#endif

#if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)||defined(PROGRAMMABLE_WITH_USB_MCHPUSB_BOOTLOADER)
	extern void _startup (void);
	#pragma code REMAPPED_RESET_VECTOR = REMAPPED_RESET_VECTOR_ADDRESS
	void _reset (void)
	{
	    _asm goto _startup _endasm
	}
#endif

#pragma code REMAPPED_HIGH_INTERRUPT_VECTOR = REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS
void Remapped_High_ISR (void)
{
	_asm goto HighPriorityISRCode _endasm
}
#pragma code REMAPPED_LOW_INTERRUPT_VECTOR = REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS
void Remapped_Low_ISR (void)
{
	_asm goto LowPriorityISRCode _endasm
}

#if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)||defined(PROGRAMMABLE_WITH_USB_MCHPUSB_BOOTLOADER)
	#pragma code HIGH_INTERRUPT_VECTOR = 0x08
	void High_ISR (void)
	{
		_asm goto REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS _endasm
	}
	#pragma code LOW_INTERRUPT_VECTOR = 0x18
	void Low_ISR (void)
	{
		 _asm goto REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS _endasm
	}
#endif

/** DECLARATIONS ***************************************************/
#pragma code

/*******************************************************************
 * x荞ݏ (Timer2荞 ADCJn)
 *******************************************************************/
#pragma interrupt HighPriorityISRCode
void HighPriorityISRCode()
{
	if (PIR1bits.TMR2IF) {
		PIR1bits.TMR2IF = 0;
		// `l؂ւ(F7ኄ荞݂ύX)
		if (ChageWait == CHANGEWAIT) {
			AdcDat[AdcChNum][BufCount] = AdcAdd;
			AdcAdd = 0;
			if (++AdcChNum == AdcRegCh) {
				AdcChNum = 0;
				if(++BufCount == ADCBUFFER) BufCount = 0;
			}
			ADCON0bits.DONE = 0;
			ADCON0bits.CHS  = AdcChannel[AdcChNum];
		}
		ChageWait--;
		if (AdcRegCh != 0 && ChageWait == 0) {
			ChageWait++;
			if (ADCON0bits.DONE) bAdcErrorFlag = TRUE;
			else {
				 ADCON0bits.GO = 1;
			}
		}
	}
}

/*******************************************************************
 * ჌x荞ݏ (ADCI荞 ADCf[^擾)
 *******************************************************************/
#pragma interruptlow LowPriorityISRCode
void LowPriorityISRCode()
{
	if (PIR1bits.ADIF) {
		PIR1bits.ADIF = 0;
		SamplCount++;
		if (AdcRegCh == 0 || SamplCount == 1) {
			return;				// 1ǂݎ̂ (Silicon Errata Ή)
		}
		AdcAdd += ((WORD)ADRESH<<8) + (WORD)ADRESL;	// I[o[TvO
		if (SamplCount == ADCSAMPLE+1) {
			SamplCount = 0;
			ChageWait = CHANGEWAIT;	// 1mS wait (Silicon Errata Ή)
			Switch2 = (Switch2<<1)+ sw2;
		}
	}
}

/*******************************************************************
 * C[v ( USB|[O / 荞݂ADCɎgp邽 )
 *******************************************************************/
void main(void)
{
	// n[hEFA
    Initialize();
    // `lo^(3`)
	EntryAnalogChannel(VCH0);
	EntryAnalogChannel(VCH1);
	EntryAnalogChannel(VCH2);
	// 荞݋
	InterruptsEnable();
	// C[v
    while(1) {
		ClrWdt();
		if (Switch2 == 0) {
			ResetToBootloader();		// u[g[_ڍs
		}
		USBDeviceTasks();				// USB^XN
		ProcessIO();					// zXgʐM
	}
}

/*******************************************************************
 * u[g[_[ڍs
 *******************************************************************/
void ResetToBootloader(void)
{
	int i;
	INTCONbits.GIEL = 0;				// ኄ荞݋֎~
	INTCONbits.GIEH = 0;				// 荞݋֎~
	for (i=0; i<500; i++) UCONbits.USBEN = 0;
	Reset();
	while(1) continue;
}

/*******************************************************************
 * 
 *******************************************************************/
void Initialize(void)
{
	ClrWdt();
	WDTCONbits.SWDTEN = 1;				// EHb`hbN^C}N
	TRISB = 0b00000000;					// RB7,RB6,(AN11),(AN10);
	TRISC = 0b11001010;					// AN9,AN8,CCP1,(RC4),AN7,CVREF,VREF-,(AN4)
	mInitAllLEDs();						// LED̏

	AdcChNum   = 0;						// `ll
	ChageWait  = 10;					// EFCgl
	SamplCount = 0;						// I[o[Tvl
	BufCount   = 0;						// f[^obt@pJE^l
	Switch2    = 0xFF;					// XCb`NA

	Vref_Setup(TRUE);					// dt@X̏
	ADC_Setup(TRUE);					// ADC̏
	Timer2_Setup(TRUE);					// Timer2̏
	Eccp1_Setup(TRUE);					// PWM̏

	InterruptsDisable();				// 荞݋֎~

    USBDeviceInit();					// USBfoCX̏
}

/*******************************************************************
 * 荞݋
 *******************************************************************/
void InterruptsEnable(void)
{
	// Interrupts Setup
	RCONbits.IPEN   = 1;				// Dxt荞ݐݒ
	INTCONbits.GIEL = 1;				// ኄ荞݋
	INTCONbits.GIEH = 1;				// 荞݋
}

/*******************************************************************
 * 荞݋֎~
 *******************************************************************/
void InterruptsDisable(void)
{
	// Interrupts Setup
	INTCONbits.GIEL = 0;				// ኄ荞݋֎~
	INTCONbits.GIEH = 0;				// 荞݋֎~
}

/*******************************************************************
 *	^C}2̐ݒ
 *******************************************************************/
void Timer2_Setup(BOOL on)
{
	if (on) { 
		// Timer2 Setup					// Fosc 48MHz
		PR2 = 59;						// (59 + 1)* (1/(Fosc/4)) = 5uS(200kHz)
		T2CONbits.T2OUTPS = 9;			// Select Postscale 1:10 (50uS Interrupt)
		T2CONbits.T2CKPS  = 1;			// Select Prescaler 1 (Fosc/4)
		PIR1bits.TMR2IF   = 0;			// TIMER2 Interrupt flag clear
		PIE1bits.TMR2IE   = 1;			// TIMER2 Interrupt enable
		IPR1bits.TMR2IP   = 1;			// TIMER2 Interrupt High Priority
		T2CONbits.TMR2ON  = 1;			// Timr2 ON
	}
	else {
		T2CONbits.TMR2ON  = 0;			// Timr2 OFF
	}
}

/*******************************************************************
 *	ECCP1̐ݒ
 *******************************************************************/
void Eccp1_Setup(BOOL on)
{
	if (on) { 
		// ECCP module Setup
		CCPR1L            = 30;			// PWM Width 50%
		CCP1CONbits.DC1B  = 0b00;		// PWM 8bit setup
		CCP1CONbits.P1M   = 0b00;		// PWM Single mode
		CCP1CONbits.CCP1M = 0b1100;		// PWM ON
		PSTRCONbits.STRA  = 1;			// CCP1 P1A output
	}
	else {
		PSTRCONbits.STRA  = 0;			// CCP1 P1A OFF
		CCP1CONbits.CCP1M = 0;			// PWM OFF
	}
}

/*******************************************************************
 * A/DRo[^̐ݒ
 *******************************************************************/
void ADC_Setup(BOOL on)
{
	// Data Initialize
	AllReentryAnalogChannel();			// `lo^̏
	if (on) { 
		// ADC Setup					// (4+11+2)*1.33uS = 22.61uS
		ADCON1bits.PVCFG0 = 0;			// selected FVR
		ADCON1bits.PVCFG1 = 1;
		ADCON1bits.NVCFG0 = 0;			// selected VSS
		ADCON1bits.NVCFG1 = 0;
		ADCON2bits.ADFM   = 1;			// right
		ADCON2bits.ACQT   = 0b010;		// 4TAD
		ADCON2bits.ADCS   = 0b110;		// Fosc/64 (TAD = 1.33uS)
		PIR1bits.ADIF     = 0;			// ADC Interrupt flag clear
		PIE1bits.ADIE     = 1;			// ADC Interrupt enable
		IPR1bits.ADIP     = 0;			// ADC Interrupt Low Priority
		ADCON0bits.ADON   = 1;			// ADC ON
	}
	else {
		ADCON0bits.ADON   = 0;			// ADC OFF
	}
}

/*******************************************************************
 * AiO`l̑So^폜
 *******************************************************************/
void AllReentryAnalogChannel(void)
{
	AdcRegCh = 0;						// ADC Channel Unregist
	TRISC |= 0xC9;						// Port C Input set
	TRISB |= 0x30;						// Port B Input set
	WPUB  |= 0x30;						// RB4/RB5 Pull-up ON
	ANSEL  = 0;							// Analog1 OFF
	ANSELH = 0;							// Analog2 OFF
}

/*******************************************************************
 * AiO`l̓o^
 *******************************************************************/
void EntryAnalogChannel(BYTE channel)
{
	if (AdcRegCh > ADCMAXCHS) return;
	switch(channel) {
	case VCH0 :							// Voltage Channel 0 
		ANSELbits.ANS7   = 1;			// AN7
		AdcChannel[AdcRegCh] = 7;
		break;
	case VCH1 :							// Voltage Channel 1
		ANSELHbits.ANS8  = 1;			// AN8
		AdcChannel[AdcRegCh] = 8;
		break;
	case VCH2 :							// Voltage Channel 2
		ANSELHbits.ANS9   = 1;			// AN9
		AdcChannel[AdcRegCh] = 9;
		break;
	case VCH3 :							// Voltage Channel 3
		ANSELHbits.ANS10 = 1;			// AN10
		AdcChannel[AdcRegCh] = 10;
		break;
	case VCH4 :							// Voltage Channel 4
		ANSELHbits.ANS11 = 1;			// AN11
		AdcChannel[AdcRegCh] = 11;
		break;
	case VCH5 :							// Voltage Channel 5
		ANSELbits.ANS4    = 1;			// AN4
		AdcChannel[AdcRegCh] = 4;
		break;
	default :
		return;
	}
	AdcRegCh++;
}

/*******************************************************************
 * dt@X̐ݒ
 *******************************************************************/
void Vref_Setup(BOOL on)
{
	if (on) {
		REFCON0bits.FVR1EN = 1;			// FVR ON
	 	REFCON0bits.FVR1S0 = 1;			// 01:1.024V / 10:2.048V / 11:4.096V
	 	REFCON0bits.FVR1S1 = 1;
		REFCON1 = 0b11101000;			// VSS-FVR / CREF ON / DAC enable
		REFCON2 = 16;					// FVR / 2 Output
	}
	else {
		REFCON1bits.D1EN   = 0;			// DAC disable
		REFCON0bits.FVR1EN = 0;			// FVR OFF
	}
}

/********************************************************************
 * ϊ
 *******************************************************************/
const static char _ADC16[] = "abcdefghijklmnop";
const static char _HEX16[] = "0123456789ABCDEF";

/********************************************************************
 * ϊ (16i̕ϊ܂B)
 *******************************************************************/
char* ByteToHex(BYTE dat, char *p)
{
	*(p++) = _HEX16[ (dat>>4)&0x0F ];
	*(p++) = _HEX16[ (dat   )&0x0F ];
	*p = 0;
	return p;
}

/********************************************************************
 * ϊ (ADC̃f[^ANԍt16i̕ϊ܂B)
 *******************************************************************/
char* AdcToHex(BYTE adcnum, WORD dat, char *p)
{
	*(p++) = _ADC16[ (adcnum)&0x0F ];
	p = ByteToHex((BYTE)(dat>>8)&0xFF, p);
	p = ByteToHex((BYTE) dat    &0xFF, p);
	return p;
}

/*******************************************************************
 * n[hEFAl[̑M
 *******************************************************************/
void Hardware_Name(void)
{
	putrsUSBUSART(HARDWARE_NAME);
}

/*******************************************************************
 * n[hEFAo[W̑M
 *******************************************************************/
void Hardware_Version(void)
{
	putrsUSBUSART(HARDWARE_VERSION);
}

/*******************************************************************
 * ADC̃rbg̑M
 *******************************************************************/
void ADC_Bits(void)
{
	ByteToHex(ADC_BITS, UsbSend);
	UsbSend[2] = '\r';
	putUSBUSART(UsbSend, 3);
}

/*******************************************************************
 * ADC̃TvȎM
 *******************************************************************/
void Number_Samples(void)
{
	ByteToHex(ADCOVRSPL, UsbSend);
	UsbSend[2] = '\r';
	putUSBUSART(UsbSend, 3);
}

/********************************************************************
 * ADC̍\
 *******************************************************************/
WORD AdcHighReso(WORD *p)
{
	int i;
	long adcl = 0;
	INTCONbits.GIEH = 0;				// 荞݋֎~
	INTCONbits.GIEL = 0;				// ኄ荞݋֎~
	for (i=0; i<ADCBUFFER; i++) {
		adcl += (long)*(p+i);
	}
	INTCONbits.GIEL = 1;				// ኄ荞݋
	INTCONbits.GIEH = 1;				// 荞݋
	return (WORD)(adcl >> ADCSHFBITS);
}

/********************************************************************
 * AiOe[^M
 *******************************************************************/
void Send_AnalogData(void)
{
	int i;
	if (AdcRegCh == 0)	return;
	for (i=0; i < AdcRegCh; i++) {
		AdcToHex(AdcChannel[i], AdcHighReso(AdcDat[i]), UsbSend+(i * ADCFORMAT));
	}
	UsbSend[i * ADCFORMAT] = '\r';
	putUSBUSART(UsbSend, (i * ADCFORMAT)+1);
}

/*******************************************************************
 * 0V(OEh)x̑M
 *******************************************************************/
void Read_Ground(void)
{
	WORD adc;
	int i;
	if (AdcRegCh == 0)	return;
	for (i=0; i<AdcRegCh; i++) {
		adc = ((WORD)Read_eeprom(EPGNDBASE+(i*2))<<8) +
			  ((WORD)Read_eeprom(EPGNDBASE+(i*2)+1));
		AdcToHex(AdcChannel[i], adc, UsbSend+(i * ADCFORMAT));
	}
	UsbSend[i * ADCFORMAT] = '\r';
	putUSBUSART(UsbSend, (i * ADCFORMAT)+1);
}

/*******************************************************************
 * 0V(OEh)x̕ۑ
 *******************************************************************/
void Write_Ground(void)
{
	WORD adc;
	int i;
	if (AdcRegCh == 0) return;
	for (i=0; i<AdcRegCh; i++) {
		WORD adc = AdcHighReso(AdcDat[i]);
		Write_eeprom(EPGNDBASE+(i*2),  (adc>>8)&0xFF);
		Write_eeprom(EPGNDBASE+(i*2)+1, adc    &0xFF);
	}
	putrsUSBUSART("OK\r");
}

/*******************************************************************
 * t@Xx̑M
 *******************************************************************/
void Read_Reference(void)
{
	WORD adc;
	int i;
	if (AdcRegCh == 0)	return;
	for (i=0; i<AdcRegCh; i++) {
		adc = ((WORD)Read_eeprom(EPREFBASE+(i*2))<<8) +
			  ((WORD)Read_eeprom(EPREFBASE+(i*2)+1));
		AdcToHex(AdcChannel[i], adc, UsbSend+(i * ADCFORMAT));
	}
	UsbSend[i * ADCFORMAT] = '\r';
	putUSBUSART(UsbSend, (i * ADCFORMAT)+1);
}

/*******************************************************************
 * t@Xx̕ۑ
 *******************************************************************/
void Write_Reference(void)
{
	WORD adc;
	int i;
	if (AdcRegCh == 0) return;
	for (i=0; i<AdcRegCh; i++) {
		WORD adc = AdcHighReso(AdcDat[i]);
		Write_eeprom(EPREFBASE+(i*2),  (adc>>8)&0xFF);
		Write_eeprom(EPREFBASE+(i*2)+1, adc    &0xFF);
	}
	putrsUSBUSART("OK\r");
}

/*******************************************************************
 * t@Xd̑M
 *******************************************************************/
void Read_VoltData(void)
{
	UsbSend[0] = Read_eeprom(EPVLTBASE+0);
	UsbSend[1] = Read_eeprom(EPVLTBASE+1);
	UsbSend[2] = Read_eeprom(EPVLTBASE+2);
	UsbSend[3] = Read_eeprom(EPVLTBASE+3);
	UsbSend[4] = '\r';
	putUSBUSART(UsbSend, 5);
}

/*******************************************************************
 * t@Xd̎MƏ
 *******************************************************************/
void Write_VoltData(void)
{
	int i=0;
	while (i < 4) {
		CDCTxService();
		if (getsUSBUSART(UsbReceive, USBMAXRCV)) {
			UsbSend[i++] = UsbReceive[0];
		}
	}
	Write_eeprom(EPVLTBASE+0, UsbSend[0]);
	Write_eeprom(EPVLTBASE+1, UsbSend[1]);
	Write_eeprom(EPVLTBASE+2, UsbSend[2]);
	Write_eeprom(EPVLTBASE+3, UsbSend[3]);
	putrsUSBUSART("OK\r");
}

/*******************************************************************
 * ADC̃`lo^̑M
 *******************************************************************/
void Channel_Number(void)
{
	int i;
	if (AdcRegCh == 0) return;
	UsbSend[0] = '0';
	UsbSend[1] = '0';
	ByteToHex(AdcRegCh, UsbSend+2);
	for (i=0; i<AdcRegCh; i++) {
		UsbSend[i+4] = _ADC16[AdcChannel[i]];
	}
	UsbSend[i+4] = '\r';
	putUSBUSART(UsbSend, i+5);
}

/********************************************************************
 * ʐM
 *******************************************************************/
void ProcessIO(void)
{
	BYTE len;
	//Blink the LEDs according to the USB device status
	BlinkUSBStatus();
	// User Application USB tasks
	if((USBDeviceState < CONFIGURED_STATE)||(USBSuspendControl==1)) return;
    if(USBUSARTIsTxTrfReady()) {
		// M
		switch (UsbReceive[0]) {
			case 's' : Send_AnalogData();	break;	// f[^M
			case 'a' : ADC_Bits();			break;	// ADCqbgM
			case 'n' : Number_Samples();	break;	// I[o[TvM
			case 'c' : Channel_Number();	break;	// `lo^M
			case 'G' : Write_Ground();		break;	// Ohx
			case 'g' : Read_Ground();		break;	// OEhx̑M
			case 'R' : Write_Reference();	break;	// t@Xx
			case 'r' : Read_Reference();	break;	// t@Xx̑M
			case 'D' : Write_VoltData();	break;	// t@Xd̏
			case 'd' : Read_VoltData();		break;	// t@Xd̑M
			case 'h' : Hardware_Name();		break;	// n[hEFAl[̑M
			case 'v' : Hardware_Version();	break;	// n[hEFAo[W̑M
			default	 : break;
		}
		UsbReceive[0] = '\0';
		// Mf[^
		if (len = getsUSBUSART(UsbReceive, USBMAXRCV)) {
			// GR[obN
			putUSBUSART(UsbReceive, len);
		}
	}
	//M
	CDCTxService();
}

/********************************************************************
 * Function:        void BlinkUSBStatus(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        BlinkUSBStatus turns on and off LEDs 
 *                  corresponding to the USB device state.
 *
 * Note:            mLED macros can be found in HardwareProfile.h
 *                  USBDeviceState is declared and updated in
 *                  usb_device.c.
 *******************************************************************/
void BlinkUSBStatus(void)
{
	static WORD led_count=0;

	if (led_count == 0)	led_count = 10000U;
	led_count--;

	#define mLED_Both_Off()		{ mYLED_Off(); mGLED_Off();}
	#define mLED_Both_On()		{ mYLED_On();  mGLED_On(); }
	#define mYLED_Only_On()		{ mYLED_On();  mGLED_Off();}
	#define mGLED_Only_On()		{ mYLED_Off(); mGLED_On(); }

	if (USBSuspendControl == 1) {
		if(led_count==0) {
			mYLED_Toggle();
			if (mGetYLED())	mGLED_On();
			else			mGLED_Off();
		}
	}
	else {
		if (USBDeviceState == DETACHED_STATE) {
			mLED_Both_Off();
		}
		else if (USBDeviceState == ATTACHED_STATE) {
			mLED_Both_On();
		}
		else if (USBDeviceState == POWERED_STATE) {
			mYLED_Only_On();
		}
		else if (USBDeviceState == DEFAULT_STATE) {
			if (led_count==0) {
				mYLED_Toggle();
				if (mGetYLED())	mGLED_Off();
				else			mGLED_On();
			}
		}
		else if (USBDeviceState == ADDRESS_STATE) {
			if (led_count == 0) {
				mYLED_Toggle();
				mGLED_Off();
			}
		}
		else if (USBDeviceState == CONFIGURED_STATE) {
			mGLED_Only_On();
		}
	}
}

// ******************************************************************************************************
// ************** USB Callback Functions ****************************************************************
// ******************************************************************************************************
// The USB firmware stack will call the callback functions USBCBxxx() in response to certain USB related
// events.  For example, if the host PC is powering down, it will stop sending out Start of Frame (SOF)
// packets to your device.  In response to this, all USB devices are supposed to decrease their power
// consumption from the USB Vbus to <2.5mA each.  The USB module detects this condition (which according
// to the USB specifications is 3+ms of no bus activity/SOF packets) and then calls the USBCBSuspend()
// function.  You should modify these callback functions to take appropriate actions for each of these
// conditions.  For example, in the USBCBSuspend(), you may wish to add code that will decrease power
// consumption from Vbus to <2.5mA (such as by clock switching, turning off LEDs, putting the
// microcontroller to sleep, etc.).  Then, in the USBCBWakeFromSuspend() function, you may then wish to
// add code that undoes the power saving things done in the USBCBSuspend() function.

// The USBCBSendResume() function is special, in that the USB stack will not automatically call this
// function.  This function is meant to be called from the application firmware instead.  See the
// additional comments near the function.

/******************************************************************************
 * Function:        void USBCBSuspend(void)
 *****************************************************************************/
void USBCBSuspend(void)
{
}

/******************************************************************************
 * Function:        void USBCBWakeFromSuspend(void)
 *****************************************************************************/
void USBCBWakeFromSuspend(void)
{
}

/********************************************************************
 * Function:        void USBCB_SOF_Handler(void)
 *******************************************************************/
void USBCB_SOF_Handler(void)
{
}

/*******************************************************************
 * Function:        void USBCBErrorHandler(void)
 *******************************************************************/
void USBCBErrorHandler(void)
{
}

/*******************************************************************
 * Function:        void USBCBCheckOtherReq(void)
 *******************************************************************/
void USBCBCheckOtherReq(void)
{
    USBCheckCDCRequest();
}

/*******************************************************************
 * Function:        void USBCBStdSetDscHandler(void)
 *******************************************************************/
void USBCBStdSetDscHandler(void)
{
    // Must claim session ownership if supporting this request
}//end

/*******************************************************************
 * Function:        void USBCBInitEP(void)
 *******************************************************************/
void USBCBInitEP(void)
{
    CDCInitEP();
}

/********************************************************************
 * Function:        void USBCBSendResume(void)
 *******************************************************************/
void USBCBSendResume(void)
{
	static WORD delay_count;
	if (USBGetRemoteWakeupStatus() == TRUE) {
		if (USBIsBusSuspended() == TRUE) {
			USBMaskInterrupts();
			USBCBWakeFromSuspend();
			USBSuspendControl = 0; 
			USBBusIsSuspended = FALSE;
			delay_count = 3600U;        
			do {
				delay_count--;
			} while(delay_count);
			USBResumeControl = 1;
			delay_count = 1800U;
			do {
				delay_count--;
			} while(delay_count);
			USBResumeControl = 0;
			USBUnmaskInterrupts();
		}
	}
}

/*******************************************************************
 * Function:        void USBCBEP0DataReceived(void)
 *******************************************************************/
#if defined(ENABLE_EP0_DATA_RECEIVED_CALLBACK)
void USBCBEP0DataReceived(void)
{
}
#endif

/*******************************************************************
 * Function:        BOOL USER_USB_CALLBACK_EVENT_HANDLER(
 *                        USB_EVENT event, void *pdata, WORD size)
 *******************************************************************/
BOOL USER_USB_CALLBACK_EVENT_HANDLER(USB_EVENT event, void *pdata, WORD size)
{
	switch (event) {
		case EVENT_TRANSFER:
			//Add application specific callback task or callback function here if desired.
			break;
		case EVENT_SOF:
			USBCB_SOF_Handler();
			break;
		case EVENT_SUSPEND:
			USBCBSuspend();
			break;
		case EVENT_RESUME:
			USBCBWakeFromSuspend();
			break;
		case EVENT_CONFIGURED: 
			USBCBInitEP();
			break;
		case EVENT_SET_DESCRIPTOR:
			USBCBStdSetDscHandler();
			break;
        case EVENT_EP0_REQUEST:
			USBCBCheckOtherReq();
			break;
		case EVENT_BUS_ERROR:
			USBCBErrorHandler();
			break;
		case EVENT_TRANSFER_TERMINATED:
			break;
		default:
			break;
	}      
	return TRUE; 
}

/** EOF main.c *************************************************/

