/*******************************************************************
 pic-lcd for 16F1937						Create Date	2013/02/12
 file name	: pic-lcd.c						Last UpDate	2013/02/28
 Complier   : HI-TECH C Ver 9.83
 
 TAB size = 4
/** Inclides *******************************************************/
#include <htc.h>
#include "GenericTypeDefs.h"
#include "t361002.h"
#include "IntenalTemp.h"

/** Configuration **************************************************/
__CONFIG(FOSC_INTOSC & WDTE_OFF & PWRTE_ON & MCLRE_OFF & BOREN_ON);
__CONFIG(LVP_OFF & PLLEN_OFF);

/** Define *********************************************************/
#define VER		"0.8"			/* t@[EFA o[V */

#define	GLED		LATDbits.LATD0	// Green LED
#define	LED_ON()	GLED = 1
#define	LED_OFF()	GLED = 0
#define	SW00		PORTEbits.RE3
#define	SW01		PORTBbits.RB3
#define	SW02		PORTBbits.RB2
#define	SW03		PORTBbits.RB1
#define	SW04		PORTDbits.RD2

#ifndef _XTAL_FREQ
 // Unless already defined assume 16MHz system frequency
 // This definition is required to calibrate __delay_us() and __delay_ms()
 #define _XTAL_FREQ 16000000
#endif

enum show_mode {
	SHW_VDD,
	SHW_TEMP,
	SHW_END
};

// EEPROM ADDRESS
#define	epCT0	0x00
#define	epCT1	0x01

/** Variables ******************************************************/
volatile BOOL	fT100;
volatile unsigned char mT0CNT = 0;
volatile unsigned char mTempCount = 100;
volatile unsigned char mShowMode = SHW_VDD;
union swinp {
	struct {
		unsigned char SWs;
	};
	struct {
		unsigned	SW0 : 1;
		unsigned	SW1 : 1;
		unsigned	SW2 : 1;
		unsigned	SW3 : 1;
		unsigned	SW4 : 1;
		unsigned	SW5 : 1;
		unsigned	SW6 : 1;
		unsigned	SW7 : 1;
	};
};
volatile union swinp mInp, mDelay;

/** Private Function ***********************************************/
void osc_16MHz(void);
void ioPortsSetup(void);
void initTimer0(void);
BOOL Timer0_irq(void);
void ShowVDD(void);
void ShowTemp(void);
void initTemp(void);
void CalibretionTrmp(void);

/** Function *******************************************************/
// 荞
static void interrupt isr(void)
{
	if (INTCONbits.TMR0IF) {
		if (Timer0_irq()) fT100 = TRUE;
	}
}

// C
void main(void)
{	
	osc_16MHz();				// 8MHz
	ioPortsSetup();				// |[g
	init_T361002();				// LCD (S_)
	__delay_ms(1000);
	lcdClear();					// S
	__delay_ms(250);
	
	mDelay.SWs = mInp.SWs =  0;	// XCb`f[^
	mShowMode = SHW_VDD;		// dd\

	initTemp();					// xZT̏
	initTimer0();				// ^C}0
	INTCONbits.GIE = 1;			// 荞݊Jn
	while (1) {
		if (fT100) {
			fT100 = FALSE;		// 100mS tONA
			ShowVDD();			// VDD \
			ShowTemp();			// x\
			// \؂ւ
			if (mInp.SW0 && !mDelay.SW0) {
				if (++mShowMode == SHW_END) mShowMode = SHW_VDD; 
			}
			mDelay.SWs = mInp.SWs;
		}
	}
}

/*******************************************************************/
// 8MHz
void osc_16MHz(void)
{								// VDD <= 2.5V
	OSCCONbits.SPLLEN = 0;		// 4xPLL OFF
	OSCCONbits.IRCF = 0b1111;	// 16MHz 
	OSCCONbits.SCS = 0b10;		// internal oscillator
}

// |[gݒ
void ioPortsSetup(void)
{
	ANSELA	= 0x00;				// All Digital
	PORTA 	= 0x00;				// '0'
	TRISA	= 0x00;				// All Output

	ANSELB	= 0x00;				// All Digital
	PORTB 	= 0x00;				// '0'
	TRISB	= 0x0E;				// Output:RB0,RB4-7 / InputRB1-3

	PORTC 	= 0x00;				// '0'
	TRISC	= 0x03;				// Output:RC2-7 / InputRC0,1

	ANSELD	= 0x00;				// All Digital
	PORTD 	= 0x00;				// '0'
	TRISD	= 0x06;				// Output:RD0,RD3-7 / InputRD1,2

	ANSELE	= 0x00;				// All Digital
	PORTE 	= 0x00;				// '0'
	TRISE	= 0x00;				// Output:RE0-2 / InputRE3
}

// Timer0 ݒ
void initTimer0(void)
{
	OPTION_REG = 0b11000111;	// Fosc/4 1:256 (8MHz//128uS)
	mT0CNT = 0;					// 100mS/(128uS*256)=3.0517....
	fT100 = FALSE;				// 3 count = 100mS..
	INTCONbits.TMR0IF = 0;
	INTCONbits.TMR0IE = 1;
}

// Timet0 荞ݏ
BOOL Timer0_irq(void)
{
	INTCONbits.TMR0IF = 0;
	if (++mT0CNT == 6) {
		mT0CNT = 0;
		mInp.SW0 = !SW00;
		mInp.SW1 = !SW01;
		mInp.SW2 = !SW02;
		mInp.SW3 = !SW03;
		mInp.SW4 = !SW04;
		return	TRUE;			// =100mS
	}
	return FALSE;				// >100mS
}

// d\
void ShowVDD(void)
{
	if (mShowMode != SHW_VDD) return;
	lcdVolt();					// d\(V)
	lcdfNumber(CheckVdd());
	mTempCount = 100;
}

// x\
void ShowTemp(void)
{
	if (mShowMode != SHW_TEMP) return;
	if (++mTempCount >= 50) {	// 5bԊu
		mTempCount = 0;
		lcdCentigrade();		// x\
		lcdtNumber(CheckTemp());
	}
	if (mInp.SW1 && !mDelay.SW1){
		mDelay.SW1 = mInp.SW1;
		CalibretionTrmp();		// x\
	}
}

// xZT
void initTemp(void)
{
	unsigned int dat = (unsigned int)(eeprom_read(epCT0)+(eeprom_read(epCT1)<<8));
	if (dat == 0xFFFF)	SetCalibretion(NOCALIBRETION);
	else				SetCalibretion((float)((int)dat)/100); 
}

// x\
void CalibretionTrmp(void)
{
	int blink = 0;
	int dat;
	while (1) {
		if (fT100) {
			fT100 = FALSE;			// 100mS tONA
			blink++;
			if (mInp.SW1 && !mDelay.SW1){
				dat = (int)(GetCalibretion()*100);
				eeprom_write(epCT0, ((unsigned int)dat&0xFF));
				eeprom_write(epCT1, ((unsigned int)dat>>8&0xFF));
				SetCalibretion((float)dat/100);
				break;
			}
			else if (blink>=6) {
				blink = 0;
				lcdClear();			// S
				lcdCentigrade();	// x\
			}
			else {
				lcdtNumber(CheckTemp());
				if (mInp.SW3 && !mDelay.SW3) {
					SetCalibretion(GetCalibretion()+0.5);
				}
				else if (mInp.SW4 && !mDelay.SW4){
					SetCalibretion(GetCalibretion()-0.5);
				}
			}
			mDelay.SWs = mInp.SWs;
		}
	}
}

/**	end of file	****************************************************/
