;==============================================================================;
;==	HID Bootloader	2KByte-Type						Create Date 2012/12/30
;==		(PIC18Non-J)								Last Update 2021/02/24
;==	file name	: main.asm									Saka Softwares
;==	Assemblr	; MPASMWIN v5.77 (MPLAB X IDE v4.20)
;==	Tab size	: 4
;==============================================================================;
;== マイクロチップ社製HID Bootloader と互換が有りWindows等のアプリが
;== そのまま使えます。かなり無理やり2KByte以内に納めています。
;== 純正Ver1.02相当/Winアプリv2.9jに対応しました。
;== 書き込みには、「HIDBootloader.exe」のv2.9j以降のWindowsアプリを
;== 使用します。以前のアプリでは、新機能が働かずブートローラーからユー
;== ザープログラムを起動できません。
;== 
;==============================================================================;
;== PWRTENとBORENを有効化に伴う弊害
;== 低電圧検知機能ですが、バスパワー動作USB機器を作る時起きます。
;== 動作中に他の機器を同一ハブに繋いだ瞬間リセットされることありました。
;== USBの電圧が規定値より下がるためです。
;== 上記ビットをOFFにすれば多少は改善されます。
;== バスパワー動作USB機器を作る場合、接続時の過大電流にも気を付けて。
;== PIC18F2xK50/45K50の問題
;== コンフィグのnLPBORがONに設定できない問題があります。
;== nLPBORをONにするには、Hexファイルを書き換える必要があります。
;== File Version ==============================================================;
#define	Version		0x0318					; BCD表現のバージョン No.	
;== 3.18	2021/02/24	イレーズ時USBの通信が止まるのを防ぐためUSBの通信タイミング
;==						を変更。ウォッチドック機能を有効にした。
;==						PIC18F2xK50/45K50の問題の対応。
;== 3.17	2018/05/03	28pin/40pinのMCLRピンの指定を変更してHexファイルを生成する
;==						のを忘れていたので自動でMCLRピンの指定できるようにした。
;==						変更のための空き領域少しし増やした。
;== 3.16	2018/04/03	バージョンをプログラムに記載するようにした。
;==						(pic16f145x同様にする)
;== 3.15	2018/02/12	アプリケーションへ移行時のリセットで 最後のUSB通信が出来な
;==						い問題を修正及びリセットまでの時間延長
;==						PIC18(L)F25K50等内部発信でUSBA使う時LEDを指定すると2KBに
;==						収まらない問題に対処（LEDの点滅処理を削除）	
;== 3.12	2016/02/11	bootloader.c EEPROMの読み出しのバグを修正
;==						及びEEPROMが無い時の処理削除位置を変更
;== 3.10	2015/11/26	Bios機能を廃止して2KBに収まるように変更
;==						C言語をXC8のみ対応します。
;==						LED対応を2つから１に
;==						誤記修正	
;== 3.01	2015/02/28	各PIC18Non-Jを統合
;== 3.00	2015/02/22	純正Ver1.02に対応及び開発環境をMPLAB Xに移行
;==						ソースファイルをUTF-8に替えています。
;==						以降旧版
;== 2.3		2013/08/20	C言語とバッティングするラベル名を変更
;== 2.2		2013/07/30	転送フラグが一部不正を修正
;== 2.1		2013/07/18	HID処理に特化する事によりサイズ縮小
;== 2.0		2013/06/24	Bios機能追加(アプリケーション側からの簡易HID処理)
;== 1.5		2013/06/15	USB(HID)通信のベクタ割り付け他
;== 1.3		2013/01/24	スタンダートリクエストのGet Statusを未サポート化他
;== 1.2		2013/01/15	後詰め処理修正/ConfigBitのPWRTENとBORENを有効化
;== 1.1		2013/01/13	未使用部分の削除及びLED機能追加
;== 1.0		2012/12/30	公開バージョン
;==		
;== Include ===================================================================;
	#include "PicConfig.inc"				; コンフィグワードを記載しています。
	#include "HardwareProfile.inc"			; ボードに合わせて修正してください。
	#include "usbhiddrv.inc"				; USB HID処理
	#include "bootloader.inc"				; プードローダ処理
	radix	dec

;== Variables =================================================================;
MAINDATA		UDATA
mCOUNT	res 3								; カウント処理用

;== Vectors ===================================================================;
POWER_ON:		CODE	0x0000
		goto	StartUp						; スタートアップ
		dw		Version						; Version (BCD code)
;==============================================================================;
;==		高割り込みベクタ
;==============================================================================;
HIGH_ISR:		CODE	0x0008
		goto	REMAPPED_HIGH_ISR_VECTOR
;==============================================================================;
;==		メインリセット (2018/05/03 分割/修正)
;==============================================================================;
Main_Reset:								; bootloader.asmから呼ばれます。
		global	Main_Reset
#ifdef BOOTLOADER_LED_ON
		bcf		pLED1						; LED1 消灯
#endif
		movlw	30							; カウンタ値
		movwf	(mCOUNT+0)
		movwf	(mCOUNT+1)
		movwf	(mCOUNT+2)
		bra		Main_Reset_Loop				; リセットへ
;==============================================================================;
;==		低割り込みベクタ
;==============================================================================;
LOW_ISR:		CODE	0x0018
		goto	REMAPPED_LOW_ISR_VECTOR
;==============================================================================;
;==		プートローダー開始アドレス (0x001C)
;==============================================================================;
BTLDR_ENTRY		CODE	BOOTLOADER_ENTRY_ADDRESS
;==============================================================================;
		bra		Bootloader					; ブートローダー へ

;== Declarations ==============================================================;
MAINCODE		CODE	0x0020
;==============================================================================;
;====	スタートアップ
;==============================================================================;
StartUp:
		clrwdt								; ウォッチドックタイマのクリア
#ifdef BOOTLOADER_SW2_ON
BootSWitch:
		btfss	pSW2						; ブートローダースイッチが押されてるか?
		bra		Bootloader
#endif
FlashSignatureCheck:						; サイン("600D")の チェック
		clrf	TBLPTRU
		movlw	high REMAPPED_RESET_VECTOR
		movwf	TBLPTRH
		movlw	low (REMAPPED_RESET_VECTOR + APP_SIGN_ADRL)
		movwf	TBLPTRL
		tblrd*+
		movf	TABLAT, W
		sublw	low  APP_SIGNATURE_VALUE
		bnz		Bootloader					; ブートローダーへ
		tblrd*+
		movf	TABLAT, W
		sublw	high APP_SIGNATURE_VALUE
		bnz		Bootloader					; ブートローダーへ
FlashRemapResetVectorCheck:				; リマップリセットベクタ検査
		movlw	low  REMAPPED_RESET_VECTOR
		movwf	TBLPTRL
		tblrd*+
		incfsz	TABLAT, W					; 0xFFでない時は
		goto	REMAPPED_RESET_VECTOR		; ユーザープログラムへ
		tblrd*+
		incfsz	TABLAT, W					; 0xFFでない時は、
		goto	REMAPPED_RESET_VECTOR		; ユーザープログラムへ
Bootloader:
		clrf	INTCON						; 割り込みの禁止
		clrf	STKPTR						; スタックポインタの初期化
		movlb	0							; Bank 0
		bsf		WDTCON, SWDTEN				; ウォッチドック起動 (2021/02/05)
#ifdef	INTERNAL_USB_CLOCK
InterNal_USB_Clock:						; 内部USBクロック使用時
		movlw	0x80
		movf	OSCTUNE						; 3X PLL ratio mode selected
		movlw	0x70
		movwf	OSCCON						; Switch to 16MHz HFINTOSC
		movlw	0x10
		movwf	OSCCON2						; Enable PLL, SOSC, PRI OSC drivers turned off
PLL_Redy_loop:							; 内部USBクロックの安定待ち
		btfss	OSCCON2, PLLRDY
		bra		PLL_Redy_loop				; Wait for PLL lock
		movlw	0x90
		movwf	ACTCON						; Enable active clock tuning for USB operation
#endif
		rcall	USBDeviceInit				; USBの初期化
		call	BootlodarInit				; ブートローダーの初期化
#ifdef BOOTLOADER_LED_ON
		bcf		pLED1						; LED1 消灯
		bcf		tLED1						; LED1 初期化
#endif
;==============================================================================;
;====	メインループ
;==============================================================================;
main:
		clrwdt								; ウォッチドックタイマのクリア
		rcall	USBDeviceTasks				; USBタスク処理
		bz		main						; 通信中以外かか?(Z=0:通信中)
#ifdef	BOOTLOADER_LED_ON
		bsf		pLED1						; LED1 点灯
#endif
main_boot:
		call	BootlodarTasks
		bra		main						; 無限ループ

;==============================================================================;
;==		メインリセット (2018/05/03 分割/修正)
;==============================================================================;
Main_Reset_Loop:
		bcf		WDTCON, SWDTEN				; ウォッチドック停止(2021/02/05)
		clrwdt								; ウォッチドックタイマのクリア
		rcall	USBDeviceTasks				; USBタスク処理
		decfsz	(mCOUNT+0), F
		bra		Main_Reset_Loop
		clrf	UCON						; USBの停止
		decfsz	(mCOUNT+1), F
		bra		Main_Reset_Loop
		decfsz	(mCOUNT+2), F
		bra		Main_Reset_Loop
		reset								; リセット

;==============================================================================;
;==		ユーザーコード領域
;==============================================================================;
SIGNATURE		CODE	APP_SIGNATURE_ADDRESS
;==============================================================================;
		dw		APP_SIGNATURE_VALUE			; ブートサインを記入
;==============================================================================;
		END