;==============================================================================;
;==	Bootloader										Create Date 2012/12/30
;==		PIC18 (Non-J Family)						Last Update 2021/02/24
;==	file name	: bootloader.asm							Saka Softwares
;==============================================================================;
;== 割り込みは一切使いません。
;== マイクロチップ社製ブートローダと互換があります。(多少違いはありますが…)
;== Ver1.02に対応しました。2Kbyte以内に収まりました。
;== 2016/02/11以前は、EEPROMの読み出しが不正になります。
;== Include ===================================================================;
#define _HID_BOOTLOADER
#include "P18CXXX.INC"
#include "usbpic18Non-J.inc"
#include "usbhiddrv.inc"
#include "bootloader.inc"

	radix	dec

;== Constants =================================================================;
#define TRUE	1
#define FALSE	0
; The bootloader version, which the bootloader PC application can do extended query to get.
; Value provided is expected to be in the format of BOOTLOADER_VERSION_MAJOR.BOOTLOADER_VERSION_MINOR
; Ex: 1.01 would be BOOTLOADER_VERSION_MAJOR == 1, and BOOTLOADER_VERSION_MINOR == 1
#define BOOTLOADER_VERSION_MAJOR	1 ; Legal value 0-255
#define BOOTLOADER_VERSION_MINOR	2 ; Legal value 0-99.  (1 = X.01)

; Section defining the address range to erase for the erase device command, along with the valid programming range to be reported by the QUERY_DEVICE command.
#define PROG_MEM_START_ADDRESS		REMAPPED_RESET_VECTOR	; Beginning of application program memory (not occupied by bootloader).  **THIS VALUE MUST BE ALIGNED WITH 64 BYTE BLOCK BOUNDRY** Also, in order to work correctly, make sure the StartPageToErase is set to erase this section.

#define QUERY_DEVICE				0x02	;Command that the host uses to learn about the device (what regions can be programmed, and what type of memory is the region)
#define UNLOCK_CONFIG				0x03	;Note, this command is used for both locking and unlocking the config bits (see the "//Unlock Configs Command Definitions" below)
#define ERASE_DEVICE				0x04	;Host sends this command to start an erase operation.  Firmware controls which pages should be erased.
#define PROGRAM_DEVICE				0x05	;If host is going to send a full RequestDataBlockSize to be programmed, it uses this command.
#define PROGRAM_COMPLETE			0x06	;If host send less than a RequestDataBlockSize to be programmed, or if it wished to program whatever was left in the buffer, it uses this command.
#define GET_DATA					0x07	;The host sends this command in order to read out memory from the device.  Used during verify (and read/export hex operations)
#define RESET_DEVICE				0x08	;Resets the microcontroller, so it can update the config bits (if they were programmed, and so as to leave the bootloader (and potentially go back into the main application)
#define SIGN_FLASH					0x09	;The host PC application should send this command after the verify operation has completed successfully.  If checksums are used instead of a true verify (due to ALLOW_GET_DATA_COMMAND being commented), then the host PC application should send SIGN_FLASH command after is has verified the checksums are as exected. The firmware will then program the SIGNATURE_WORD into flash at the SIGNATURE_ADDRESS.
#define QUERY_EXTENDED_INFO			0x0C	;Used by host PC app to get additional info about the device, beyond the basic NVM layout provided by the query device command

; Unlock Configs Command Definitions
#define UNLOCKCONFIG				0x00	;Sub-command for the ERASE_DEVICE command
#define LOCKCONFIG					0x01	;Sub-command for the ERASE_DEVICE command

; Query Device Response "Types" 
#define MEMORY_REGION_PROGRAM		0x01	;When the host sends a QUERY_DEVICE command, need to respond by populating a list of valid memory regions that exist in the device (and should be programmed)
#define MEMORY_REGION_EEDATA		0x02
#define MEMORY_REGION_CONFIG		0x03
#define MEMORY_REGION_USERID		0x04
#define MEMORY_REGION_END			0xFF	;Sort of serves as a "null terminator" like number, which denotes the end of the memory region list has been reached.
#define BOOTLOADER_NEWER_FLAG		0xA5	;Tacked on in the VersionFlag byte, to indicate when using newer version of bootloader with extended query info available

; BootState Variable States
#define IDLE						0x00
#define NOT_IDLE					0x01

; OtherConstants
#define INVALID_ADDRESS				0xFFFFFFFF
#define CORRECT_UNLOCK_KEY			0xB5	; 未使用

; Application and Microcontroller constants
#define BYTES_PER_ADDRESS_PIC18		0x01	;One byte per address.	PIC24 uses 2 bytes for each address in the hex file.
#define USB_PACKET_SIZE				0x40
#define WORDSIZE					0x02	;PIC18 uses 2 byte words, PIC24 uses 3 byte words.
#define RequestDataBlockSize		0x3A	;Number of data bytes in a standard request to the PC.	Must be an even number from 2-58 (0x02-0x3A).  Larger numbers make better use of USB bandwidth and 
											;yeild shorter program/verify times, but require more micrcontroller RAM for buffer space.
;#define USB_PACKET_SIZE					0x40

; General command (with data in it) packet structure used by PROGRAM_DEVICE and GET_DATA commands		
#define Command						0	; char
#define Address						1	; long
#define Adr8						1	; cahr (Address)	& 0xFF
#define Adr16						2	; char (Address>>8) & 0xFF
#define Adr24						3	; char (Address>>16)& 0xFF
#define Adr32						4	; char (Address>>24)& 0xFF
#define Size						5	; char
#define Datas						6	; char [REQUEST_DATA_BLOCK_SIZE]
; This struct used for responding to QUERY_DEVICE command (on a device with four programmable sections)
;#define	Command					0	; char
#define PacketDataFieldSize			1	; char
#define BytesPerAddress				2	; char
#define Type1						3	; char
#define Address1					4	; long
#define Length1						8	; long
#define Type2						12	; char
#define Address2					13	; long
#define Length2						17	; long
#define Type3						21	; char
#define Address3					22	; long
#define Length3						26	; long
#define Type4						30	; char
#define Address4					31	; long
#define Length4						35	; long
#define Type5						39	; char
#define Address5					40	; long
#define Length5						44	; long
#define Type6						48	; char
#define Address6					49	; long
#define Length6						53	; long
#define VersionFlag					57	; char
#define ExtraPadBytes				58	; dhar [6]
; For UNLOCK_CONFIG command
;#define	Command					0	; char
#define LockValue					1	; char
; Structure for the QUERY_EXTENDED_INFO command (and response)
;#define	Command					0	; char
#define BootloaderVersion			1	; int
#define ApplicationVersion			3	; int
#define AppVerLow					3	; char (ApplicationVersion)    & 0xFF
#define AppVerHigh					4	; char (ApplicationVersion>>8) & 0xFF
#define SignatureAddress			5	; long
#define SignatureValue				9	; int
#define ErasePageSize				11	; int
#define Config1LMask				13	; char
#define Config1HMask				14	; char
#define Config2LMask				15	; char
#define Config2HMask				16	; char
#define Config3LMask				17	; char
#define Config3HMask				18	; char
#define Config4LMask				19	; char
#define Config4HMask				20	; char
#define Config5LMask				21	; char
#define Config5HMask				22	; char
#define Config6LMask				23	; char
#define Config6HMask				24	; char
#define Config7LMask				25	; char
#define Config7HMask				26	; char

;== Variables =================================================================;
#define BOOT_RAM_ADR	0x100			; Bank1を使用
BOOTBUF		UDATA	BOOT_RAM_ADR
mPGBUF		res ERASE_PAGE_SIZE			; Programing Buffer
mPKTToPC	res USB_PACKET_SIZE			; PacketToPC
mPKTFrPC	res USB_PACKET_SIZE			; PacketFromPC

mBKBSR		res 1						; Backup BSR
mBTSTSTE	res 1						; BootState
mCFGLKVAL	res 1						; ConfigsLockValue
mPRGPNT		res 3						; 書き込みポインタ
mCOUNT		res 1						; プログラムカウンタ
mPRGCOMP	res 1						; 書き込み完了フラグ
mWBCNT		res 1						; ライトブロックカウンタ

;==============================================================================;
DEVICEDATA:	CODE_PACK
;==============================================================================;
;== PICのデバイス情報 
#define PROG_MEM_LENGTH  PROG_MEM_STOP_ADDRESS - PROG_MEM_START_ADDRESS
DeviceDatas:
	db	QUERY_DEVICE							; Command
	db	RequestDataBlockSize					; PacketDataFieldSize
	db	BYTES_PER_ADDRESS_PIC18					; BytesPerAddress

	db	MEMORY_REGION_PROGRAM					; Type1
	dw	(PROG_MEM_START_ADDRESS)&0xFFFF
	dw	(PROG_MEM_START_ADDRESS>>16)&0xFFFF		; Address1
	dw	(PROG_MEM_LENGTH)&0xFFFF
	dw	(PROG_MEM_LENGTH>>16)&0xFFFF			; Length1

	db	MEMORY_REGION_CONFIG					; Type2
	dw	(CONFIG_START_ADDRESS)&0xFFFF
	dw	(CONFIG_START_ADDRESS>>16)&0xFFFF		; Address2
	dw	(CONFIG_SECTION_LENGTH)&0xFFFF
	dw	(CONFIG_SECTION_LENGTH>>16)&0xFFFF		; Length2

	db	MEMORY_REGION_USERID					; Type3
	dw	(USER_ID_ADDRESS)&0xFFFF
	dw	(USER_ID_ADDRESS>>16)&0xFFFF			; Address3
	dw	(USER_ID_SIZE)&0xFFFF
	dw	(USER_ID_SIZE>>16)&0xFFFF				; Length3

#ifdef DEVICE_WITH_EEPROM
	db	MEMORY_REGION_EEDATA					; Type4
	dw	(EEPROM_EFFECTIVE_ADDRESS)&0xFFFF
	dw	(EEPROM_EFFECTIVE_ADDRESS>>16)&0xFFFF	; Address4
	dw	(EEPROM_SIZE)&0xFFFF
	dw	(EEPROM_SIZE>>16)&0xFFFF				; Length4
#endif

	db	MEMORY_REGION_END						; Type4or5
DeviceDatas_END:
;== 拡張情報 (Bootloder Ver1.02以降)
ExtendedInfo:
	db	QUERY_EXTENDED_INFO						; Command
	db	BOOTLOADER_VERSION_MINOR
	db	BOOTLOADER_VERSION_MAJOR				; BootloaderVersion
	dw	0x00									; ApplicationVersion (後で)
	dw	(APP_SIGNATURE_ADDRESS)&0xFFFF
	dw	(APP_SIGNATURE_ADDRESS>>16)&0xFFFF		; SignatureAddress
	dw	APP_SIGNATURE_VALUE						; SignatureValue
	dw	(ERASE_PAGE_SIZE)&0xFFFF
	dw	(ERASE_PAGE_SIZE>>16)&0xFFFF			; ErasePageSize
	db	0xFF									; Config1LMask
	db	0xFF									; Config1HMask
	db	0xFF									; Config2LMask
	db	0xFF									; Config2HMask
	db	0x00									; Config3LMask
	db	0xFF									; Config3HMask
	db	0xFF									; Config4LMask
	db	0x00									; Config4HMask
	db	0xFF									; Config5LMask
	db	0xFF									; Config5HMask
	db	0xFF									; Config6LMask
	db	0xFF									; Config6HMask
	db	0xFF									; Config7LMask
	db	0xFF									; Config7HMask
ExtendedInfo_END:

;== Declarations ==============================================================;
BOOTCODE:	CODE
;==============================================================================;
;==		プートーローダー初期化部
;==============================================================================;
BootlodarInit:
		global	BootlodarInit
		clrf	WREG					; バンク対応
		movff	WREG, mBTSTSTE			; IDLE
		movlw	TRUE
		movff	WREG, mCFGLKVAL			; TRUE
		movff	WREG, mPRGCOMP			; TRUE
		return
;== BootlodarInit end

;==============================================================================;
;==		ブートローダー実行部
;==============================================================================;
BootlodarTasks:
		global	BootlodarTasks
		movff	BSR, mBKBSR				; BSR保存
		movlb	high BOOT_RAM_ADR		; ブート用バンク
		movf	mBTSTSTE, F
		bnz		Boot_Working			; ブートローダー処理中
;==============================================================================;
;==		受信待ち処理
;==============================================================================;
Boot_Idle:
		rcall	HIDRxIsBusy				; 受信したか?
		bnz		Boot_Idle_Exit			; -- return (2)
		movlb	high USB_RAM_ADR		; USB用バンク
		movlw	low  (mPKTFrPC)
		movwf	mHIDRxADRL
		movlw	high (mPKTFrPC)
		movwf	mHIDRxADRH
		movlw	USB_PACKET_SIZE
		movwf	mHIDRxCNT
		movlw	fPIBUSY
		movwf	mHIDRxFLAG
		rcall	HIDRxReport				; データ取得
		movlb	high BOOT_RAM_ADR		; ブート用バンク
		incf	mBTSTSTE, F				; NOT_IDLE
		movlw	USB_PACKET_SIZE
		lfsr	FSR0, mPKTToPC
BOOT_TxBUF_Clear:
		clrf	POSTINC0				; 送信データクリア
		decfsz	WREG, F
		bra		BOOT_TxBUF_Clear
;== 戻り位置(1)
Boot_NotIdle_Exit:
		movff	mBKBSR, BSR				; BSR復帰
		return

;==============================================================================;
;==		受信処理
;==============================================================================;
Boot_Working:
		movf	mPKTFrPC, W				; 0x00
										; -- Unknown --
		decf	WREG, F					; 0x01
										; -- Unknown --
		dcfsnz	WREG, F					; 0x02
		bra		Query_Device			; QUERY_DEVICE
		dcfsnz	WREG, F					; 0x03
		bra		Unlock_Config			; UNLOCK_CONFIG
		dcfsnz	WREG, F					; 0x04
		bra		Erase_Device			; ERASE_DEVICE
		dcfsnz	WREG, F					; 0x05
		bra		Program_Device			; PROGRAM_DEVICE
		dcfsnz	WREG, F					; 0x06
		bra		Program_Complete		; PROGRAM_COMPLETE
		dcfsnz	WREG, F					; 0x07
		bra		Get_Data				; GET_DATA
		dcfsnz	WREG, F					; 0x08
		bra		Reset_Device			; RESET_DEVICE
		dcfsnz	WREG, F					; 0x09
		bra		Sign_Flash				; SIGN_FLASH
		decf	WREG, F					; 0x0A
										; -- Unknown --
		decf	WREG, F					; 0x0B
										; -- Unknown --
		dcfsnz	WREG, F					; 0x0C
		bra		Query_Extended_Info		; QUERY_EXTENDED_INFO
;== 戻り位置(2)
Boot_Idle_Exit:
		clrf	mBTSTSTE				; IDLE
		movff	mBKBSR, BSR				; BSR復帰
		return

;==============================================================================;
;==	ROMデータをRAMバッファに転送
;==============================================================================;
Move_RomToRam:
		tblrd*+
		movf	TABLAT, W
		movwf	POSTINC0				; データ転送
		decfsz	mCOUNT, F
		bra		Move_RomToRam
		return

;==============================================================================;
;== サイン記入処理 (Ver1.02以降)
;==============================================================================;
Sign_Flash:
		clrf	TBLPTRU
		movlw	high (PROG_MEM_START_ADDRESS)
		movwf	TBLPTRH
		movlw	low  (PROG_MEM_START_ADDRESS)
		movwf	TBLPTRL					; 転送元
		lfsr	FSR0, mPGBUF			; 転送先
		movlw	ERASE_PAGE_SIZE
		movwf	mCOUNT					; 転送数
		rcall	Move_RomToRam			; データ転送
Sign_Flash_Set_Code:
		movlw	low  (APP_SIGNATURE_VALUE)
		movwf	mPGBUF + APP_SIGN_ADRL
		movlw	high (APP_SIGNATURE_VALUE)
		movwf	mPGBUF + APP_SIGN_ADRH	; サイン記入
Sign_Flash_BlockErase:
;;		clrf	TBLPTRU
;;		movlw	high (PROG_MEM_START_ADDRESS)
;;		movwf	TBLPTRH					; イレーズサイズが64以下のため
		movlw	low  (PROG_MEM_START_ADDRESS)
		movwf	TBLPTRL
		rcall	ProgramFlashErase		; イレーズ
Sign_Flash_WriteFlash:
		tblrd*-							; dummy read decrement
		lfsr	FSR0, mPGBUF			; 転送元
		movlw	(ERASE_PAGE_SIZE / WRITE_BLOCK_SIZE)
		movwf	mCOUNT					; 転送ブロック数
Sign_Flash_SetWriteBlockSize:
		movlw	WRITE_BLOCK_SIZE
		movwf	mWBCNT					; ブロックサイズ
Sign_Flash_Write_Loop:
		movf	POSTINC0, W
		movwf	TABLAT
		tblwt+*							; データ転送
		decfsz	mWBCNT, F
		bra		Sign_Flash_Write_Loop
		rcall	ProgramFlashWrite		; プログラム領域に書き込み
		decfsz	mCOUNT, F
		bra		Sign_Flash_SetWriteBlockSize
		bra		Boot_Idle_Exit			; -- return (2)

;==============================================================================;
;== 拡張情報の問い合わせ処理 (Ver1.02以降)
;==============================================================================;
Query_Extended_Info:
		rcall	HIDTxIsBusy				; 送信可能か?
		btfss	STATUS, Z
		bra		Boot_NotIdle_Exit		; -- return (1)
		clrf	TBLPTRU
		movlw	high  (ExtendedInfo)
		movwf	TBLPTRH
		movlw	low   (ExtendedInfo)
		movwf	TBLPTRL					; 転送元
		lfsr	FSR0, mPKTToPC			; 転送先
		movlw	(ExtendedInfo_END - ExtendedInfo)
		movwf	mCOUNT					; 転送数
		rcall	Move_RomToRam			; データ転送
		clrf	TBLPTRU
		movlw	high (APP_VERSION_ADDRESS)
		movwf	TBLPTRH
		movlw	low  (APP_VERSION_ADDRESS)
		movwf	TBLPTRL
		tblrd*+
		movf	TABLAT, W
		movwf	(mPKTToPC + AppVerLow)	; アプリバージョン(MINOR)を記入
		tblrd*+
		movf	TABLAT, W
		movwf	(mPKTToPC + AppVerHigh) ; アプリバージョン(MAJOR)を記入
		rcall	Send_PaketData			; パケット送信
		bra		Boot_Idle_Exit			; -- return (2)

;==============================================================================;
;== デバイス情報の問い合わせ処理
;==============================================================================;
Query_Device:
		rcall	HIDTxIsBusy				; 送信可能か?
		btfss	STATUS, Z
		bra		Boot_NotIdle_Exit		; -- return (1)
		clrf	TBLPTRU
		movlw	high  (DeviceDatas)
		movwf	TBLPTRH
		movlw	low   (DeviceDatas)
		movwf	TBLPTRL					; 転送元
		lfsr	FSR0, mPKTToPC			; 転送先
		movlw	(DeviceDatas_END - DeviceDatas)
		movwf	mCOUNT					; 転送数
		rcall	Move_RomToRam			; データ転送
		movlw	BOOTLOADER_NEWER_FLAG
		movwf	(mPKTToPC + VersionFlag); Ver1.02以降に対応
		rcall	Send_PaketData			; パケット送信
		bra		Boot_Idle_Exit			; -- return (2)

;==============================================================================;
;==		ロック/アンロック処理
;==============================================================================;
Unlock_Config:
		movlw	TRUE
		movwf	mCFGLKVAL				; TRUE
		movlw	UNLOCKCONFIG
		cpfseq	(mPKTFrPC + LockValue)
		bra		Boot_Idle_Exit			; -- return (2)
		movlw	FALSE
		movwf	mCFGLKVAL				; FALSE
		bra		Boot_Idle_Exit			; -- return (2)

;==============================================================================;
;==		イレーズ処理
;==============================================================================;
Erase_Device:
		clrf	TBLPTRU
		movlw	high (PROG_MEM_START_ADDRESS)
		movwf	TBLPTRH
		movlw	low  (PROG_MEM_START_ADDRESS)
		movwf	TBLPTRL
Prog_Erase_Loop:
		rcall	ProgramFlashErase		; プログラム領域の消去
		movlw	ERASE_PAGE_SIZE
		addwf	TBLPTRL, F
		btfsc	STATUS, C
		incf	TBLPTRH, F				; 次の消去ブロックアドレス
		movlw	low  (PROG_MEM_STOP_ADDRESS)
		cpfseq	TBLPTRL
		bra		Prog_Erase_Loop
		movlw	high (PROG_MEM_STOP_ADDRESS)
		cpfseq	TBLPTRH
		bra		Prog_Erase_Loop
EEPROM_Erase:
#ifdef DEVICE_WITH_EEPROM
		clrf	EEADR
EEPROM_Erase_Loop:
		movlw	0xFF
		movwf	EEDATA					; 0xFF
		rcall	EepromErase				; EEPROM領域の消去
		incf	EEADR, F
		movlw	EEPROM_SIZE &0xFF		; Max 256Byte
		cpfseq	EEADR
		bra		EEPROM_Erase_Loop
#endif
UserID_Erase:
		movlw	upper (USER_ID_ADDRESS)
		movwf	TBLPTRU
		clrf	TBLPTRH
		clrf	TBLPTRL
		rcall	UserIdErase				; UserID領域の消去
		bra		Boot_Idle_Exit			; -- return (2)

;==============================================================================;
;==		ブログラム処理
;==============================================================================;
Program_Device:
		movff	(mPKTFrPC + Size), mCOUNT	; 転送サイズ
		lfsr	FSR0, mPKTFrPC + Datas	; 転送データ
		movf	(mPKTFrPC + Size), W
		sublw	(USB_PACKET_SIZE - Datas)
		addwf	FSR0L, F
		btfsc	STATUS, C
		incf	FSR0H					; 後詰め処理
		movlw	0xF0					; EEPROM Address
		cpfseq	(mPKTFrPC + Adr24)
		bnz		Program_FlashROM
Write_EEPROM:
#ifdef DEVICE_WITH_EEPROM				; 修正 2016/02/11
		movff	(mPKTFrPC + Adr8), EEADR
Write_EEPROM_Loop:
		movff	POSTINC0, EEDATA
		rcall	EepromWrite				; EEPROMへ書き込み
		incf	EEADR, F				; EEPROM Address +1
		decfsz	mCOUNT, F
		bra		Write_EEPROM_Loop
#endif									; 修正 2001/02/11
		bra		Boot_Idle_Exit			; -- return (2)
Program_FlashROM:
		movff	(mPKTFrPC + Adr8),	TBLPTRL
		movff	(mPKTFrPC + Adr16), TBLPTRH
		movff	(mPKTFrPC + Adr24), TBLPTRU
		movlw	0x30
		cpfseq	(mPKTFrPC + Adr24)
		bra		Write_FlashProg
		movf	mCFGLKVAL, F
		bz		Write_ConfigBits		; Unlockの時
		bra		Boot_Idle_Exit			; -- return (2) Error
Write_ConfigBits:
		movff	POSTINC0, TABLAT		; データ転送
		tblwt*
		rcall	ConfigBitsWrite			; コンフィグビットの書き込み
		tblrd*+							; ダミーリード
		decfsz	mCOUNT, F
		bra		Write_ConfigBits
		bra		Boot_Idle_Exit			; -- return (2)

Write_FlashProg:
		movf	mPRGCOMP, F
		bnz		Write_FlashProg_Start	; 書き込み開始 (mPRGCOMP == TRUE)
		movf	(mPRGPNT + 0), W
		cpfseq	TBLPTRL
		bra		Boot_Idle_Exit			; -- return (2) Error
		movf	(mPRGPNT + 1), W
		cpfseq	TBLPTRH
		bra		Boot_Idle_Exit			; -- return (2) Error
		movf	(mPRGPNT + 2), W
		cpfseq	TBLPTRU
		bra		Boot_Idle_Exit			; -- return (2) Error
		bra		Write_FlashProg_ReStart ; 追加書き込み
Write_FlashProg_Start:
		clrf	mPRGCOMP				; 書込開始(FALSE)
Write_FlashProg_Loop:
		movf	POSTINC0, W
		movwf	TABLAT
		tblwt*+
		decf	mCOUNT, F
		bz		Write_FlashProg_Exit	; データが無くなった
Write_FlashProg_ReStart:
		movf	TBLPTRL, W
		andlw	(WRITE_BLOCK_SIZE - 1)	; 書き込みブロックサイズ
		bnz		Write_FlashProg_Loop
Write_FlashProg_WR:
		tblrd*-							; ダミーリード
		rcall	ProgramFlashWrite		; プログラム領域に書き込み
		tblrd*+							; ダミーリード
		movf	mCOUNT, F
		bnz		Write_FlashProg_Loop
Write_FlashProg_Exit:
		movff	TBLPTRL, (mPRGPNT + 0)
		movff	TBLPTRH, (mPRGPNT + 1)
		movff	TBLPTRU, (mPRGPNT + 2)
		bra		Boot_Idle_Exit			; -- return (2)

;==============================================================================;
;==		プログラム完了処理
;==============================================================================;
Program_Complete:
		movlw	TRUE
		movwf	mPRGCOMP				; 書込終了(TRUE)
		movff	(mPRGPNT + 0), TBLPTRL
		movff	(mPRGPNT + 1), TBLPTRH
		movff	(mPRGPNT + 2), TBLPTRU
		clrf	mCOUNT
		bra		Write_FlashProg_WR

;==============================================================================;
;==		データ読出し処理
;==============================================================================;
Get_Data:
		rcall	HIDTxIsBusy				; 送信可能か?
		btfss	STATUS, Z
		bra		Boot_NotIdle_Exit		; -- return (1)
		movlw	GET_DATA
		movwf	(mPKTToPC + Command)	; GET_DATA コマンド
		movff	(mPKTFrPC + Adr8),	(mPKTToPC + Adr8)
		movff	(mPKTFrPC + Adr16), (mPKTToPC + Adr16)
		movff	(mPKTFrPC + Adr24), (mPKTToPC + Adr24)
		movff	(mPKTFrPC + Adr32), (mPKTToPC + Adr32)	; 転送アドレス
		movf	(mPKTFrPC + Size), W
		movwf	(mPKTToPC + Size)		; 転送サイズ
		movwf	mCOUNT
		lfsr	FSR0, (mPKTToPC + Datas)	; 転送データ
		movf	(mPKTFrPC + Size), W
		sublw	(USB_PACKET_SIZE - Datas)
		addwf	FSR0L, F
		btfsc	STATUS, C
		incf	FSR0H					; 後詰め処理
		movlw	0xF0					; EEPROM Address
		cpfseq	(mPKTToPC + Adr24)
		bnz		Read_FlashROM			; 修正 2016/02/11(bra->bnz)
Read_EEPROM:
#ifdef DEVICE_WITH_EEPROM				; 修正 2016/02/11
		movf	(mPKTToPC + Adr8), W
		movwf	EEADR
Read_EEPROM_Loop:
		clrf	EECON1					; 修正 2016/02/11
		bsf		EECON1, RD				; EEPROMのデータ読み込み設定
		movff	EEDATA, POSTINC0
		incf	EEADR, F				; EEPROM Address +1
		decfsz	mCOUNT, F
		bra		Read_EEPROM_Loop
#endif									; 修正 2016/02/16
		bra		Get_Data_SendData
Read_FlashROM:
		movff	(mPKTToPC + Adr8),	TBLPTRL
		movff	(mPKTToPC + Adr16), TBLPTRH
		movff	(mPKTToPC + Adr24), TBLPTRU
Read_FlashROM_Loop:
		tblrd*+
		; since 0x300004 and 0x300007 are not implemented we need to return 0xFF
		; since the device reads 0x00 but the hex file has 0x00
		movlw	0x30
		cpfseq	(mPKTToPC + Adr24)
		bra		Read_FlashROM_SetData
		movlw	0x05
		subwf	TBLPTRL, W
		bz		Read_FlashROM_0xFF
		movlw	0x08
		subwf	TBLPTRL, W
		bz		Read_FlashROM_0xFF
		bra		Read_FlashROM_SetData
Read_FlashROM_0xFF:
		movlw	0xFF
		movwf	TABLAT
		;//
Read_FlashROM_SetData:
		movff	TABLAT, POSTINC0
		decfsz	mCOUNT
		bra		Read_FlashROM_Loop
Get_Data_SendData:
		rcall	Send_PaketData			; パケット送信
		bra		Boot_Idle_Exit			; -- return (2)

;==============================================================================;
;==		リセット処理	(ユーザープログラムへ移行)
;==============================================================================;
Reset_Device:
		movff	mBKBSR, BSR				; BSR復帰
		goto	Main_Reset				; メインへ

;==============================================================================;
;==		内部処理
;==============================================================================;
;==		パケットデータの送信
;==============================================================================;
Send_PaketData:
		movlb	high (USB_RAM_ADR)		; USB用バンク
		movlw	low  (mPKTToPC)
		movwf	mHIDTxADRL
		movlw	high (mPKTToPC)
		movwf	mHIDTxADRH
		movlw	USB_PACKET_SIZE			; 64Byte
		movwf	mHIDTxCNT
		movlw	fPIRAM | fPIBUSY
		movwf	mHIDTxFLAG
		rcall	HIDTxReport				; データ送信
		movlb	high BOOT_RAM_ADR		; ブート用バンク
		return

;==============================================================================;
;==		各メモリ書き換え実行部
;==============================================================================;
;== EEPROMの消去
EepromErase:
;		clrwdt							; ウォッチドックタイマのクリア
;		rcall	USBDeviceTasks			; USBタスク処理	(削除 2021/01/31)	
		movlw	b'00000100'				; EEPROMの消去設定
		bra		Set_EECON1
;== UserIDの消去
UserIdErase:
		movlw	b'10010100'				; UserIDの消去設定
		bra		Set_EECON1
;== フラッシュメモリの消去
ProgramFlashErase:
;		clrwdt							; ウォッチドックタイマのクリア
;		rcall	USBDeviceTasks			; USBタスク処理 	(削除 2021/01/31)
		movlw	b'10010100'				; プログラムメモリの消去設定
		bra		Set_EECON1
;== EEPROMの書き込み
EepromWrite:
		movlw	b'00000100'				; EEPROMのデータ書込設定
		bra		Set_EECON1
;== コンフィグビットの書き込み(UserID?)
ConfigBitsWrite:
		movlw	b'11000100'				; ConfigBitの書込設定
		bra		Set_EECON1
;== フラッシュメモリの書き込み
ProgramFlashWrite:
		movlw	b'10100100'				; Flash Programの書込設定
;== EECONの書き込み
Set_EECON1:
		movwf	EECON1
		clrf	INTCON					; 割り込みの禁止(念のため)
		clrwdt							; ウォッチドックタイマのクリア(2021/02/05)
Memery_Write:
		movlw	0x55
		movwf	EECON2
		movlw	0xAA	
		movwf	EECON2
		bsf		EECON1, WR
Memery_Write_wait:
		btfsc	EECON1, WR
		bra		Memery_Write_wait		
		bcf		EECON1, WREN			; 書き換え時間 約2mS
		return

;==============================================================================;
		END
