;==============================================================================;
;==	PIC USB HID Driver								Create Date 2013/06/16
;==		for PIC18 (Non-J Family)					Last Update 2018/02/11
;==	file name	: usbhiddrv.asm								Saka Softwares
;==============================================================================;
;== USB処理は、ポーリング専用です。割り込みは一切使いません。
;== 2KByteに対応するため、サブルーチン部分を統合して読み難くなっています。
;== USB 操作部分は全てUSB RAM上で行なっていますので、Bank情報に気を付けてください。
;== usbclassとusbdrvを統合しました。
;== Include ===================================================================;
	#define _PIC18F_USB_DRIVER
	#include "P18CXXX.INC"
	#include "usbpic18Non-J.inc"
	#include "usbhiddrv.inc"
	#include "usbdesc.inc"

	radix	dec

;== Constants =================================================================;
#define		USBBANK high(USB_RAM_ADR)	; USB RAMアドレスの上位
; DBレジスタ名化
#define		BDTMAXADR	mBD3+3			; BDTの最後のアドレス
; Buffer Descriptor 0
#define		BD0STAT		mBD0+0			; EP0 OUT (受信処理用)
#define		BD0CNT		mBD0+1
#define		BD0ADRL		mBD0+2
#define		BD0ADRH		mBD0+3
; Buffer Descriptor 1
#define		BD1STAT		mBD1+0			; EP0 IN (送信処理用)
#define		BD1CNT		mBD1+1
#define		BD1ADRL		mBD1+2
#define		BD1ADRH		mBD1+3
; Buffer Descriptor 2
#define		BD2STAT		mBD2+0			; EP1 OUT (受信処理用)
#define		BD2CNT		mBD2+1
#define		BD2ADRL		mBD2+2
#define		BD2ADRH		mBD2+3
; Buffer Descriptor 3
#define		BD3STAT		mBD3+0			; EP1 IN (送信処理用)
#define		BD3CNT		mBD3+1
#define		BD3ADRL		mBD3+2
#define		BD3ADRH		mBD3+3
; BDnSTAT Bits
UOWN		EQU			H'007'			; オーナーフラグ 0:CPU/1:SIE
DTS			EQU			H'006'			; 0:DAT0/1:DAT1
DTSEN		EQU			H'003'			; CPU時 DTS イネーブルを書き込む
BSTALL		EQU			H'002'			; CPU時 バッファストールを書き込む
BC9			EQU			H'001'			; 0 (64Byte以下の転送のため必ず0)
BC8			EQU			H'000'			; 0 (64Byte以下の転送のため必ず0)
PID3		EQU			H'005'			; TRNIF割り込み後
PID2		EQU			H'004'			; このトークンPID値でも
PID1		EQU			H'003'			; 通信の状態を確認する事ができる
PID0		EQU			H'002'			; (下記参照)
; BDnSTAT flags
#define		fSIE		0x80			; SIE
#define		fCPU		0x00			; CPU
#define		fDAT1		0x40			; DTS = 1
#define		fDAT0		0x00			; DTS = 0
#define		fDTSEN		0x08			; DTSイネーブル
#define		fBSTALL		0x04			; バッファストール
; トークン PID 下記3種を判別 (その他のPID値は省略)
#define		PID_OUT		B'0001'<<2		; 0x1 OUT token
#define		PID_IN		B'1001'<<2		; 0x9 IN token
#define		PID_SOF		B'0101'<<2		; 0x5 SOF token　（未使用）
#define		PID_SETUP	B'1101'<<2		; 0xD SETUP token
#define		PID_BITS	B'00111100'		; DBnSTATのPIDビット

; セットアップ状態フラグ
#define WAIT_SETUP		0				; セットアップ待ち状態
#define CTRL_TRF_TX		1				; コントロール送信状態
#define CTRL_TRF_RX		2				; コントロール受信状態

;== Variables =================================================================;
USBRAM		UDATA USB_RAM_ADR			; USB BD Address
; バッファ ディスクリプタ テーブル (ハードウェアによりRAM位置固定)
mBD0			res 4					; Buffer Descriptor 0
mBD1			res 4					; Buffer Descriptor 1
mBD2			res 4					; Buffer Descriptor 2
mBD3			res 4					; Buffer Descriptor 3
; USB 通信用バッファ
bmRequestType	res 1					; セットアップパケット
bRequest		res 1					;	 〃
wValue			res 2					;	 〃
wIndex			res 2					;	 〃
wLength			res 2					;	 〃
#if (EP0_BUF_SIZE-8)
				res EP0_BUF_SIZE-8		; 8Byte以上設定の時
#endif
#define BD0BUF		bmRequestType		; BD0通信バッファ
BD1BUF			res EP0_BUF_SIZE		; BD1通信バッファ
BD2BUF			res EP1_BUF_SIZE		; BD2通信バッファ
BD3BUF			res EP1_BUF_SIZE		; BD3通信バッファ

;==============================================================================;
; USB RAMの余りを使用する (BDTと通信バッファを含め223(0xDF)byte以下にする事)
mOLDBSR			res 1					; バンクの保存用
mFSR0			res 2					; FSR0 保存用
mFSR1			res 2					; FSR1 保存用
mFSR2			res 2					; FSR2 保存用
; 標準ディスクリプタ データ保存用
mUSBCNFG		res 1					; USB Configuration の保存用
mUSBIF0			res 1					; USB Interface0 の保存用
mUSBIF1			res 1					; USB Interface1 の保存用
mIDLERATE		res 1					; USB Idle Rate の保存用
mPROTOCOL		res 1					; USB Protocol の保存用
; 通信制御用
#define mRCVCNT		BD0CNT				;　受信データ数
mSNDCNT			res 1					; 送信データ数
mMEMCMT			res 1					; メモリ間転送数
mBD0STAT		res 1					; BD0STAT仮設定用
mUSTAT			res 1					; USB STATUS reg 保存用
mFIFOCNT		res 1					; USTAT用FIFOカウント用
; USB 状態フラグ
mUSBSTAE		res 1					; USBDeviceState
; セットアップ状態フラグ
mCTRLST			res 1					; controlTransferState

;==============================================================================;
PIPERAM		UDATA USB_RAM_ADR+0xEC		; アドレス固定
; 通信用パイプ
mCODE									; ブート移行コード
PIPE0o			res 5					; EP0 Out pipe
PIPE0i			res 5					; EP0 In  pipe
PIPE1o									; EP1 Out pipe
mRxFLG			res 1					; 受信制御フラグ
mRxCNT			res 1					; 受信数
mRxADR			res 2					; 転送先 RAM Address
mRxADRU			res 1					; Reserved
PIPE1i									; EP1 In  pipe
mTxFLG			res 1					; 送信制御フラグ
mTxCNT			res 1					; 送信数
mTxADR			res 2					; 転送元 RAM Address
mTxADRU			res 1					; 転送元 ROM Upper Address

;== Variables Global ==========================================================;
	global	PIPE0o, PIPE0i, PIPE1o, PIPE1i

;== Declarations ==============================================================;
USB_CODE:	CODE
;==============================================================================;
;==		USB 初期化部
;==============================================================================;
USBDeviceInit:
		global	USBDeviceInit			; USBの初期化 (ラベル名変更)
		clrf	UIE						; USBの各割り込みを無効化して
		movff	UIE, UEIE				; USBエラー割り込みも無効化する(バンク対応)
		clrf	UCON					; USB制御レジスタをクリア
USB_ModuleInit:
		movlw	b'00010100'				; USBプルアップ抵抗/フルスピードに
		movwf	UCFG					; 設定する
		rcall	USB_BDT_Clear			; BDTをクリア
USB_ModuleInit_Wait1:
		bsf		UCON, USBEN				; USBモジュールを有効にする。
		btfss	UCON, USBEN
		bra		USB_ModuleInit_Wait1	; 適切なクロックが供給されるまで待つ
USB_ModuleInit_Wait2:
		btfsc	UCON, SE0				; シングルエンド検出下か?
		bra		USB_ModuleInit_Wait2
		clrf	UIR						; 全ての割り込みフラグをクリア
		movlw	POWERED_STATE
		movff	WREG, mUSBSTAE			; USB電源投入状態を設定(バンク対応)
		return
;== _usbDeviceInit end
;==============================================================================;
;==		USB デバイスタスク処理部
;==============================================================================;
USBDeviceTasks:
		global	USBDeviceTasks
		btfss	UCON, USBEN
		return							; USBモジュールが無効
;== USB アクティビティフラグ
		btfsc	UIR, ACTVIF 
		rcall	USB_Activity			; ウェイクアップ処理
;== サスペンド中
		btfsc	UCON, SUSPND
		bra		USBDeviceTasks_Exit		; サスペンドなら終了
;== USB バス リセットフラグ
		btfsc	UIR, URSTIF
		rcall	USB_BusReset			; 初期化処理
;== USB アイドルフラグ
		btfsc	UIR, IDLEIF
		rcall	USB_BusIdle				; サスペンド処理
;== USB ストールフラグ
		btfsc	UIR, STALLIF
		rcall	USB_BusStall			; バスストール処理
;== 通信完了処理
		btfsc	UIR, TRNIF
		rcall	USB_EP0Service			; EP0通信処理
;== 通信完了の終了処理
USBDeviceTasks_Exit:
		bcf		STATUS, Z				; 通信状態			Z=0
		movlw	CONFIGURED_STATE
		cpfseq	mUSBSTAE
		bsf		STATUS, Z				; セットアップ状態	Z=1
		movff	mUSBSTAE, WREG			; 戻り値
		return
;== USBDeviceTasks end
;==============================================================================;
;==		EP1受信中フラグの取得
;==============================================================================;
HIDRxIsBusy:
		global	HIDRxIsBusy
		movff	BD2STAT, WREG			; バンク対応
		andlw	b'10000000'				; UOWNbitの抜き出し
		return							; Z=0:SIE / Z=1:CPU
;== HIDRxIsBusy end
;==============================================================================;
;==		EP1送信中フラグの取得
;==============================================================================;
HIDTxIsBusy:
		global	HIDTxIsBusy
		movff	BD3STAT, WREG			; バンク対応
		andlw	b'10000000'				; UOWNbitの抜き出し
		return							; Z=0:SIE / Z=1:CPU
;== HIDTxIsBusy end
;==============================================================================;
;==		データ(EP1)受信処理
;==============================================================================;
HIDRxReport:
		global	HIDRxReport
		lfsr	FSR0, BD2BUF			; 受信バッファアドレスをセット
		lfsr	FSR1, mPI1oFLAG			; パイプフラグアドレスをセット 
		movf	BD2CNT, W				; 受信数
		rcall	USB_OutBufSet			; データ転送
		rcall	USB_EP1oDTS_Toggle		; 次の受信処理
		return
;== HIDRxReport end
;==============================================================================;
;==		データ(EP1)送信処理
;==============================================================================;
HIDTxReport:
		global	HIDTxReport
		btfss	mPI1iFLAG, bPIBUSY
		return							; 送信データが無い
		lfsr	FSR0, mPI1iBUF			; 送信バッファアドレスをセット
		lfsr	FSR1, mPI1iFLAG			; パイプフラグアドレスをセット 
		movlw	EP1_BUF_SIZE			; バッファサイズ
		call	USB_INBufSet			; バッファ転送
		rcall	USB_EP1iDTS_Toggle		; 送信
		return
;== HIDTxReport end
;==============================================================================;
;== 内部処理 ルーチン
;== 各処理モジュールです。
;==============================================================================;
;==		使用BDTのクリア
;==============================================================================;
USB_BDT_Clear:
		clrf	BD0STAT					; BD0STATのクリア
		clrf	BD1STAT					; BD1STATのクリア
		clrf	BD2STAT					; BD2STATのクリア
		clrf	BD3STAT					; BD3STATのクリア
		return

;==============================================================================;
;==		バスアクティビリティ処理
;==============================================================================;
USB_Activity:
;;		rcall	ウェイクアップユーザー処理			; 未サポート
		bcf		UCON, SUSPND			; サスペンドの解除
USB_Activity_loop
		btfss	UIR, ACTVIF
		return							; 確実にクリアして抜ける
		bcf		UIR, ACTVIF				; フラグクリア
		bra		USB_Activity_loop

;==============================================================================;
;==		バスリセット処理
;==============================================================================;
USB_BusReset:
		BANKSEL UEP0					;　バンク切り換え
		clrf	UEP0					; UEP0のクリア
		clrf	UEP1					; UEP1のクリア
		clrf	UADDR					; USB アドレス0の設定
		movlb	USBBANK					; USB RAMに戻す
		rcall	USB_EP0_FirstSetup		; エンドポイント0を初期状態化に設定
		bcf		UCON, SUSPND			; サスペンドの解除
		clrf	mUSBCNFG				; USB configurationの初期化
		bcf		UIR, URSTIF				; クリアフラグ
		movlw	DEFAULT_STATE
		movwf	mUSBSTAE				; USB初期状態
		return

;==============================================================================;
;==		バスアイドル処理
;==============================================================================;
USB_BusIdle:
		bsf		UCON, SUSPND			; サスペンドへ移行
;;		rcall	サスペンドユーザー処理			; 未サポート
		bcf		UIR, IDLEIF				; フラグクリア
		return

;==============================================================================;
;==		バスストール処理
;==============================================================================;
USB_BusStall:
		BANKSEL UEP0					; EPバンク切り換え
		btfss	UEP0, EPSTALL
		bra		USB_BusStall_Exit
		movlb	USBBANK					; USB RAMに戻す
		btfss	BD0STAT, UOWN			; EP0oをSIE使用中
		bra		USB_BusStall_Exit
		btfss	BD1STAT, UOWN			; EP0iをSIE使用中
		bra		USB_BusStall_Exit
		btfss	BD1STAT, BSTALL			; EP0iがストール中
		bra		USB_BusStall_Exit
		rcall	USB_EP0oStall			; Setupパケット待ちにする
		BANKSEL UEP0					; EPバンク切り換え
		bcf		UEP0, EPSTALL			; EP0のストール解除
USB_BusStall_Exit:
		movlb	USBBANK					; USB RAMに戻す
		bcf		UIR, STALLIF			; フラグクリア
		return

;==============================================================================;
;== コントロール通信
;==============================================================================;
USB_EP0Service:
		movf	USTAT, W
		movwf	mUSTAT					; ステータスを保存
		bcf		UIR, TRNIF				; フラグクリア
		movlw	0x38
		andwf	mUSTAT, W				; EP0か調べる
		btfss	STATUS, Z
		return							; EP0以外の時
		btfsc	mUSTAT, DIR				; INトークン?
		bra		USB_EP0Input			; 送信処理
		movf	BD0STAT, W				; EP0oのステータスを読み出す。
		andlw	PID_BITS				; PIDビットを抜き出す。
		sublw	PID_SETUP				; PIDはセットアップか?
		bnz		USB_EP0Output			; 受信処理
;==============================================================================;
;== セットアップコントロールの転送 ([n]は、DAT番号)
;== Control Write:
;== <SETUP[0]><OUT[1]><OUT[0]>...<IN[1]> | <SETUP[0]> (HID-bootloaderでは発生しない)
;== Control Read:
;== <SETUP[0]><IN[1]><IN[0]>... <OUT[1]> | <SETUP[0]>
;== Control Onry:
;== <SETUP[0]><IN[1]> | <SETUP[0]>
;== 必ずSETUPパケットはDAT0できます。
;== 最後に、データサイズ0でDAT1の返答パケットがあります。(デバイスからも出す。)
;== 未サポートのセットアップが発生したらストールして次のセットアップ待ちになります。
;== 通常時セットアップパケットが来たらパイプをフラッシュしてセットアップに備えます。
;==============================================================================;
;== セットアップ受信処理
;==============================================================================;
USB_EP0Setup:
		clrf	mCTRLST					; Set WAIT_SETUP(0)
		;	EP0iの制御用BDTを強制的にCPUオーナーにする	(無いと通信エラーになる)	
		bcf		BD1STAT, UOWN			; EP0iのBDTをCPUオーナーにする
		;	通常の通信を止める為パイプをフラッシュする
		clrf	mPI0iFLAG				; パイプ0in フラグクリア
		clrf	mPI0iCNT				; パイプ0in カウントグクリア
		clrf	mPI0oFLAG				; パイプ0out フラググクリア
		clrf	mPI0oCNT				; パイプ0out カウントグクリア
		;	各セットアップ受信処理
		rcall	USBCheckStdRequest		; スタンダートリクエスト
		rcall	USBCheckClassRequest	; クラスリクエスト
;==============================================================================;
;==		セットアップ後処理
;==============================================================================;
USB_EP0Setup_Exit:
		bcf		UCON, PKTDIS			; パケット転送の許可
		btfsc	mPI0iFLAG, bPIBUSY		; 送信フラグ
		bra		USB_Setup_input			; 送信処理へ
		rcall	USB_EP0oStall			; セットアップ受信待ち
		rcall	USB_EP0iStall			; 送信禁止送信
		return							; 終了

;==============================================================================;
;==		セットアップデータ送信部
;==============================================================================;
USB_Setup_input:
		btfss	bmRequestType, 7		; セットアップでデバイスからホストへ転送
		bra		USB_HostToDevice
;==============================================================================;
;==		デバイスからホストへ データ送信する
;==============================================================================;
USB_DeviceToHost:						; ホストへ送信
		movlw	CTRL_TRF_TX
		movwf	mCTRLST					; コントロール送信処理中
		movf	(wLength+1), W
		bnz		USB_DeviceToHost_ACK	; 256Bye以上転送可能
		movf	mPI0iCNT, W				; 送信予定数(255byte以下)
		subwf	(wLength+0), W			; 要求転送数
		btfss	STATUS, C				; 要求転送数 < 送信予定数の時
		movff	(wLength+0), mPI0iCNT	; 要求転送数に変更
;==============================================================================;
;==		返答受信設定部
;==============================================================================;
USB_DeviceToHost_ACK:
; <SETUP[0]><IN[1]><IN[0]>......<OUT[1]> | <SETUP[0]>
; CTRL_TRF_TX フラグが立っていたら ↑このパケットを受信する
		rcall	USB_EP0iBufSet			; <IN[1]]>
		rcall	USB_EP0iDTS_1			; DTSを1にして送信
		rcall	USB_EP0oDTS_All			; <OUT[1]>返答パケットを受信
		return							; 終了

;==============================================================================;
;==		ホストからデバイスへ データ送信された
;==============================================================================;
USB_HostToDevice:
		movlw	CTRL_TRF_RX
		movwf	mCTRLST					; コントロール受信処理中
;==============================================================================;
;==		返答送信部
;==============================================================================;
USB_HostToDevice_ACK:
; <SETUP[0]><OUT[1]><OUT[0]>...<IN[1]> | <SETUP[0]>
;					 <SETUP[0]><IN[1]> | <SETUP[0]>
; CTRL_TRF_RX フラグが立っていたら↑このパケットを送信する
		rcall	USB_EP0oStall			; 次のセットアップ受信待ち
		clrf	mSNDCNT					; 0Byteのデータ
		rcall	USB_EP0iDTS_1			; <IN[1]>返答パケットを送信
		return							; 終了

;==============================================================================;
;== コントロール受信処理
;==============================================================================;
USB_EP0Output:
		rcall	USB_EP0oStall			; 次のセットアップパケット待ち
		clrf	mCTRLST					; Set WAIT_SETUP(0)
		return							; 終了

;==============================================================================;
;== コントロール送信処理
;==============================================================================;
USB_EP0Input:
;== USBアドレスの設定
		movlw	ADR_PENDING_STATE
		cpfseq	mUSBSTAE				; アドレスペンディングステート?
		bra		USB_CTIH_Tx
		movff	(wValue+0), UADDR		; USB Addressを設定
		movf	(wValue+0), W			; 0か調べる
		btfss	STATUS, Z
		movlw	ADDRESS_STATE			; USB Addressが0の時
		btfsc	STATUS, Z
		movlw	DEFAULT_STATE			; USB Addressが0以外の時
		movwf	mUSBSTAE				; ステートを変更する
;== データ送信
USB_CTIH_Tx:
		movlw	CTRL_TRF_TX
		cpfseq	mCTRLST					; 受信中の送信なら
		bra		USB_CTIH_Tx_Done		; 返答パケット送信処理へ
		btfss	mPI0iFLAG, bPIBUSY		; 送信フラグが立っていない時は
		return							; 終了
		rcall	USB_EP0iBufSet			; 送信データ転送処理
		rcall	USB_EP0iDTS_Toggle		; DTSを反転して送信
		return							; 終了
;== 返答パケット送信完了
USB_CTIH_Tx_Done:
		bcf		mPI0oFLAG, bPIBUSY
		clrf	mCTRLST					; Set WAIT_SETUP(0)
		return							; 終了

;==============================================================================;
;==		スタンダートリクエスト処理部
;==============================================================================;
USBCheckStdRequest:
		movf	bmRequestType, W
		andlw	B'01100000'				; Mask Type Bit
		bz		StandardRequest			; '00' スタンダートリクエスト
		return
StandardRequest:
		movf	bRequest, W				; 0x00
;;		bz		GetStatus				; Get Status		(未サポート)
		decf	WREG, F					; 0x01
;;		bra		ClearFeature			; Clear Feature		(未サポート)
		decf	WREG, F					; 0x02
										; -- Unknown --
		decf	WREG, F					; 0x03
;;		bra		SetFeature				; Set Feature		(未サポート)
		decf	WREG, F					; 0x04
										; -- Unknown --
		dcfsnz	WREG, F					; 0x05
		bra		SetAddress				; Set Address
		dcfsnz	WREG, F					; 0x06
		bra		GetDescriptor			; Get Descriptor
		decf	WREG, F					; 0x07
;;		bra		SetDescriptor			; Set Descriptor	(未サポート)
		dcfsnz	WREG, F					; 0x08
		bra		GetConfiguration		; Get Configuration
		dcfsnz	WREG, F					; 0x09
		bra		SetConfiguration		; Set Configuration
;;		dcfsnz	WREG, F					; 0x0A
;;		bra		GetInterface			; Get Interface		(未サポート)
;;		dcfsnz	WREG, F					; 0x0B
;;		bra		SetInterface			; Set Interface		(未サポート)
;;		dcfsnz	WREG, F					; 0x0C
;;		bra		SynchFrame				; Synch Frame		(未サポート)
		return

;==============================================================================;
;==		デバイスステータスの送信
;==============================================================================;
;GetStatus:
;		clrf	(mPI0iBUF+0)			; 送信データに
;		clrf	(mPI0iBUF+1)			; 0を入れる
;		movf	bmRequestType, W
;		andlw	b'00000011'
;		bz		GetStatusDevise			; 0:Device
;		dcfsnz	WREG, F
;		bra		GetStatusInterface		; 1:Interface
;		dcfsnz	WREG, F
;		bra		GetStatusEndPiont		; 2:Endpoint
;		return							; Error
;== デバイス情報
;GetStatusDevise:
;		[0]: bit0: Self-Powered Status [0] Bus-Powered [1] Self-Powered
;			 bit1: RemoteWakeup		   [0] Disabled    [1] Enabled
;		movlw	CFG_ATTRIBUTES
;		btfsc	WREG, 6					; Self-Powered Bit
;		bsf		(mPI0iBUF+0), 0			; b0を1にする
;		btfsc	WREG, 5					; RemoteWakeup Bit
;		bsf		(mPI0iBUF+0), 1			; b1を1にする
;		bra		GST_exit
;== インターフェス情報
;GetStatusInterface:
;		[0]: bit0: Halt Status [0] Not Halted [1] Halted
;		フォルトしないので処理無し
;		bra		GST_exit
;== エンドポイント情報
;GetStatusEndPiont:
;		lfsr	FSR0, BD0STAT			; BDnSTATのベースアドレス
;		movlw	b'10001111'				; b7:OUT/IN b3-0;EP No. (b'd000nnnn')
;		andwf	(wIndex+0), W			; を抜き出す。
;		swapf	WREG, F					; 上位下位の入れ換え(b'nnnnd000')
;		rrncf	WREG, F					; エンドポイントNoを1bit右シフトして(b'0nnnnd00')
;		addwf	FSR0L, F				; 加算するとFSR0に BDnSTATアドレスが入る
;		movlw	low(BDTMAXADR)
;		cpfslt	FSR0L					; BDnSTATに設定されていない時は
;		return							; Error
;		btfss	INDF0, UOWN				; SIEが使用中で
;		bra		GST_exit
;		btfsc	INDF0, BSTALL			; ストールしてるなら
;		bsf		(mPI0iBUF+0), 0			; 送信データを0x01にする
;;		bra		GST_exit
;== 各ステータスの送信
;GST_exit:
;		movlw	2
;		movwf	mPI0iCNT				; データは2byte
;		bra		Request_RAM_Send		; - return

;==============================================================================;
;==		特定機能のクリア
;==============================================================================;
;ClearFeature:
;		return							; (未サポート)

;==============================================================================;
;==		特定機能の設定
;==============================================================================;
;SetFeature:
;		return							; (未サポート)

;==============================================================================;
;==		デバイスアドレスの取得
;==============================================================================;
SetAddress:
		movlw	ADR_PENDING_STATE		; アドレスペンディングへ
		movwf	mUSBSTAE				; 移行(送信時に書き込み)
		bra		Request_Answer_Send		; - return

;==============================================================================;
;==		各ディスクプリタの送信
;==============================================================================;
GetDescriptor:
		btfss	bmRequestType, 7		; デバイスからホストに転送か?
		return
		clrf	mPI0iADRU				; 転送元アドレス(Upper)
;== ディスクプリタの選択
		decf	bDscType, W				; 0x01
		bz		GDC_DeviceDescriptor
		dcfsnz	WREG, F					; 0x02
		bra		GDC_ConfigurationDescriptor
		dcfsnz	WREG, F					; 0x03
		bra		GDC_StringDescriptor
;;		dcfsnz	WREG, F					; 0x04
;;		bra		GDC_IntefaceDescriptor	; (未使用)
;;		dcfsnz	WREG, F					; 0x05
;;		bra		GDC_EndPointDescriptor	; (未使用)
		return
;== デバイスディスクリプタの送信
GDC_DeviceDescriptor:
		movlw	low(DEVDSC01)			; デバイスディスクリプタ
		movwf	mPI0iADRL				; 転送元アドレス(Low)
		movlw	high(DEVDSC01)
		movwf	mPI0iADRH				; 転送元アドレス(High)
		movlw	DEVDSC_SIZE
		movwf	mPI0iCNT				; 送信データ数を設定
		bra		Request_ROM_Send		; - return
;== コンフィクレーションディスクリプタの送信
GDC_ConfigurationDescriptor:
		movlw	low(CFGDSC01)
		movwf	mPI0iADRL				; 転送元アドレス(Low)
		movlw	high(CFGDSC01)
		movwf	mPI0iADRH				; 転送元アドレス(High)
		movlw	CFGDSC01_SIZE
		movwf	mPI0iCNT				; 送信データ数を設定
		bra		Request_ROM_Send		; - return
;== ストリイグディスクリプタの選択
GDC_StringDescriptor:
		movwf	bDscIndex, W			; 0x00
		bz		GDC_LanguageString
		dcfsnz	WREG, F					; 0x01
		bra		GDC_ManufacturerString
		dcfsnz	WREG, F					; 0x02
		bra		GDC_ProductString
;;		dcfsnz	WREG, F					; 0x03
;;		bra		GDC_SerialNumString		; (未使用)
;;		dcfsnz	WREG, F					; 0x04
;;		bra		GDC_ConfigurationString ; (未使用)
;;		dcfsnz	WREG, F					; 0x05
;;		bra		GDC_InterfaceString		; (未使用)
		return
;== ランゲージストリングディスクリプタの送信
GDC_LanguageString:
		movlw	low(LANGDSC)			; 言語ディスクリプタ
		movwf	mPI0iADRL				; 転送元アドレス(Low)
		movlw	high(LANGDSC)
		movwf	mPI0iADRH				; 転送元アドレス(High)
		movlw	LANGDSC_SIZE
		movwf	mPI0iCNT
		bra		Request_ROM_Send		; - return
;== 製造者名ディスクリプタの送信
GDC_ManufacturerString:
		movlw	low(MFGDSC01)			; 製造社名
		movwf	mPI0iADRL				; 転送元アドレス(Low)
		movlw	high(MFGDSC01)
		movwf	mPI0iADRH				; 転送元アドレス(High)
		movlw	MFGDSC01_SIZE
		movwf	mPI0iCNT				; 送信データ数を設定
		bra		Request_ROM_Send		; - return
;== 製品名ディスクリプタの送信
GDC_ProductString:
		movlw	low(PRDDSC01)			; 製品名
		movwf	mPI0iADRL				; 転送元アドレス(Low)
		movlw	high(PRDDSC01)
		movwf	mPI0iADRH				; 転送元アドレス(High)
		movlw	PRDDSC01_SIZE
		movwf	mPI0iCNT				; 送信データ数を設定
		bra		Request_ROM_Send		; - return

;==============================================================================;
;==		セットディスクプリタ
;==============================================================================;
;SetDescriptor:
;		return							; (未サポート)

;==============================================================================;
;==		コンフィグレーションの送信
;==============================================================================;
GetConfiguration:
		movff	mUSBCNFG, (mPI0iBUF+0)
		incf	mPI0iCNT, F				; データは1byte
		bra		Request_RAM_Send		; - return

;==============================================================================;
;==		コンフィグレーションの取得
;==============================================================================;
SetConfiguration:
		movf	bConfigValue, W
		movwf	mUSBCNFG				; USB Configurationを保存
		bz		SetConfig_clrear
SetConfig_Set:
		movlw	b'00011110'				; ハンドシェイク,コントロール無効,OUT/INを
		movff	WREG, UEP1				; EP1に設定 (バンク対応)

		rcall	USB_EP1oInit			; EP1 OUT 初期化
		rcall	USB_EP1iInit			; EP1 IN 初期化

		movlw	CONFIGURED_STATE		; コンフィグレーション終了を
		movwf	mUSBSTAE				; 書き込み
		bra		Request_Answer_Send		; - return
SetConfig_clrear:
		movlw	ADDRESS_STATE			; アドレスステートを先ずセット
		movwf	mUSBSTAE				; ステータス書き込み
		bra		Request_Answer_Send		; - return

;==============================================================================;
;==		代替インターフェース情報の送信
;==============================================================================;
;GetInterface:
;		movff	mUSBIF0, (mEP0iBUF+0)	; IF0を送信バッファへ
;		btfsc	(wIndex+0), 0			; IF numberが1なら
;		movff	mUSBIF1, (mEP0iBUF+0)	; IF1を送信バッファに
;		movlw	1
;		movwf	mPI0iCNT				; データは1byte
;		bra		Request_RAM_Send		; - return

;==============================================================================;
;==		代替インターフェース情報の取得
;==============================================================================;
;SetInterface:
;		btfss	(wIndex+0), 0			; IF numberが0なら
;		movff	(wValue+0), mUSBIF0		; IF0に保存へ
;		btfsc	(wIndex+0), 0			; IF numberが1なら
;		movff	(wValue+0), mUSBIF1		; IF1に保存
;		bra		Request_Answer_Send		; - return

;==============================================================================;
;==		シンクフレーム
;==============================================================================;
;SynchFrame;
;		return							; (未サポート)

;==============================================================================;
;====	クラスリクエスト
;==============================================================================;
USBCheckClassRequest:
		global	USBCheckClassRequest
;== HID クラスリクエスト(一部省略しています。)
USBCheckHIDRequest:
		movf	bmRequestType, W
		andlw	b'00011111'				; Device,Interface,Endpoint,Other
		sublw	RCPT_INTF				; Interface
		bz		HID_RequestIFnumber
		return
HID_RequestIFnumber:
		movlw	HID_INTF_ID				; Interface Number
		cpfseq	bIntfID
		return
HID_RequestGetDescriptor:
		movlw	GET_DESCRIPTOR			; Get Descriptor
		cpfseq	bRequest				; をリクエスト?
		bra		ClassRequest
		movlw	DSC_HID
		subwf	bDscType, W				; DSC_HID : 0x21
;;		bz		HID_Request_HID			; (未サポート)
		decf	WREG, F					; DSC_RPT : 0x22
		bz		HID_Request_RPT
;;		decf	WREG, F					; DSC_PHY : 0x23
;;		bz		HID_Request_PHY			; (未サポート)
		bra		ClassRequest
;HID_Request_HID:
;		movlw	1
;		cpfseq	mUSBCNFG				; USBコンフィグが終了しているか?
;		bra		ClassRequest
;		movlw	low(HIDDSC01)
;		movwf	mPI0iADRL				; 転送元アドレス(Low)
;		movlw	high(HIDDSC01)
;		movwf	mPI0iADRH				; 転送元アドレス(High)
;		movlw	HIDDSC_SIZE
;		movwf	mPI0iCNT				; データは9byte
;		bra		Request_ROM_Send		; - return (ROMから送信)
HID_Request_RPT:
		movlw	low(HIDRPT01)
		movwf	mPI0iADRL				; 転送元アドレス(Low)
		movlw	high(HIDRPT01)
		movwf	mPI0iADRH				; 転送元アドレス(High)
		movlw	HIDRPT01_SIZE
		movwf	mPI0iCNT				; データ数設定
		bra		Request_ROM_Send		; - return (ROMから送信)
;HID_Request_PHY:
;		return							; (未サポート)
ClassRequest:
;		movwf	bmRequestType, W
;		andlw	b'01100000'				; 
;		sublw	(CLASS<<5)				; ClassRequest
;		bz		ClassDescription
		return							; (未サポート)
;ClassDescription:
;		movlw	0x08
;		cpfslt	bRequest				; 0x08より小さい時スキップ
;		bra		ClassREQ_Set
;ClassREQ_Get:
;		decf	bRequest, W				; 0x01
;;		bz		GetReport				; GetReport (未サポート)
;		dcfsnz	WREG, F					; 0x02
;		bra		GetIdle
;		dcfsnz	WREG, F					; 0x03
;		bra		GetProtocol
;		return
;ClassREQ_Set:
;		movf	bRequest, W
;		bcf		WREG, 3
;		decf	WREG, F					; 0x09
;;		bz		SetReport				; SetReport (未サポート)
;		dcfsnz	WREG, F					; 0x0A
;		bra		SetIdle
;		dcfsnz	WREG, F					; 0x0B
;		bra		SetProtocol
;		return
;GetIdle:
;		movff	mIDLERATE, (mPI0iBUF+0) ; データ転送
;		movlw	1
;		movwf	mPI0iCNT				; データは1byte
;		bra		Request_RAM_Send		; - return (RAMから送信)
;SetIdle:
;		movff	(wValue+1), mIDLERATE
;		bra		Request_Answer_Send		; - return (返答パケット送信)
;GetProtocol:
;		movff	mPROTOCOL, (mPI0iBUF+0) ; データ転送
;		movlw	1
;		movwf	mPI0iCNT				; データは1byte
;		bra		Request_RAM_Send		; - return (RAMから送信)
;SetProtocol:
;		movff	(wValue+0), mPROTOCOL
;		bra		Request_Answer_Send		; - return (返答パケット送信)

;==============================================================================;
;==		送信処理指定 (usbClass.asm も使用)
;==============================================================================;
Request_RAM_Send:
		movlw	fPIRAM|fPIBUSY|fPINCPY	; 転送バッファにコピー済み
		movwf	mPI0iFLAG				; RAMから送信
		return

Request_ROM_Send:
		movlw	fPIROM|fPIBUSY
		movwf	mPI0iFLAG				; ROMから送信
		return

Request_Answer_Send:
		bsf		mPI0iFLAG, bPIBUSY		; 返答パケット送信
		return

;==============================================================================;
;==		共通受信バッファ転送部
;==============================================================================;
USB_OutBufSet:
		movf	PREINC1, F				;					(mPInoCNT)
		bz		USB_OBufRead_Exit		; 転送先バッファが0Byteの時
		movf	WREG, W					; 受信したデータ数
		bz		USB_OBufRead_Exit		; 受信数が0Byeき時
		cpfslt	INDF1					; mPI0oCNT < mRCVCNT
		bra		USB_OBufRead_Count
		movf	INDF1, W				; 受信予定数だけ取得する
USB_OBufRead_Count:
		subwf	POSTINC1, F				; 受信予定数の残り		(mPInoCNT)
		movff	POSTINC1, FSR2L			; 転送先アドレス(Low)	(mPInoADRL)
		movff	INDF1,	  FSR2H			; 転送先アドレス(High)	(mPInoADRH)
USB_OBufRead_Loop:
		movff	POSTINC0, POSTINC2		; 受信データ転送
		decfsz	WREG, F
		bra		USB_OBufRead_Loop
		movff	FSR2H, POSTDEC1			;					(mPInoADRH)
		movff	FSR2L, INDF1			; 次の転送先アドレス		(mPInoADRL)
USB_OBufRead_Exit:
		return

;==============================================================================;
;==		コントロール(EP0)送信バッファ転送部
;==============================================================================;
USB_EP0iBufSet:
		lfsr	FSR0, mPI0iBUF			; 送信バッファアドレスをセット
		lfsr	FSR1, mPI0iFLAG			; パイプフラグアドレスをセット 
		movlw	EP0_BUF_SIZE			; バッファサイズ
;==============================================================================;
;==		共通送信バッファ転送部
;==============================================================================;
USB_INBufSet:
		movwf	mSNDCNT					; 送信バッファサイズ
		cpfslt	PREINC1					; mPIniCNT < EPn_BUF_SIZE
		bra		USB_INBuf_SetCount
		movff	INDF1, mSNDCNT			; 全てのデータを送信		(mPIniCNT)
USB_INBuf_SetCount:
		movf	mSNDCNT, W				; 今回の送信数
		subwf	POSTDEC1, F				; 今回残った送信数 (mPIniCNTCNT - mSNDCNT)
		movf	mSNDCNT, W
		bnz		USB_INBuf_RomOrRam		; 1Byte以上送信
		bra		USB_INBuf_ClearBusy		; 0byte送信
USB_INBuf_RomOrRam:
		movwf	mMEMCMT					; メモリ間転送数
		btfsc	INDF1, bPIRAM			; 転送元のデータが		(mPIniFLAG)
		bra		USB_INBuf_Ram			; RAM上にデータが有る		
USB_INBuf_Rom:
		movlw	2
		addwf	FSR1L, F
		movff	POSTINC1, TBLPTRL		; ROM転送元アドレス		(mPIniADRL)
		movff	POSTINC1, TBLPTRH		;					(mPIniADRH)
		movff	INDF1,	  TBLPTRU		;					(mPIniADRU)
USB_INBuf_RomLoop:
		tblrd*+
		movff	TABLAT, POSTINC0		; データ転送
		decfsz	mMEMCMT, F
		bra		USB_INBuf_RomLoop
		movff	TBLPTRU, POSTDEC1		;					(mPIniADRU)
		movff	TBLPTRH, POSTDEC1		;					(mPIniADRH)
		movff	TBLPTRL, POSTDEC1		; 次のROM転送元アドレス (mPIniADRL)
		bra		USB_INBuf_CheckCount
USB_INBuf_Ram:
		btfsc	POSTINC1, fPINCPY
		bra		USB_INBuf_CheckCount	; 既にデータ転送済み
		incf	FSR1L, F
		movff	POSTINC1, FSR2L			;					(mPIniADRL)
		movff	INDF1,	  FSR2H			; RAM転送元アドレス		(mPIniADRH)
USB_INBuf_RamLoop:
		movff	POSTINC2, POSTINC0		; データ転送 
		decfsz	mMEMCMT, F
		bra		USB_INBuf_RamLoop
		movff	FSR1H, POSTDEC1			;					(mPIniADRH)
		movff	FSR1L, POSTDEC1			; 次のRAM転送元アドレス (mPIniADRL)
USB_INBuf_CheckCount:
		movf	POSTDEC1, F				; 次に送るデータがあるか?	(mPIniCNT)
		btfsc	STATUS, Z
USB_INBuf_ClearBusy:
		bcf		INDF1, bPIBUSY			; 全て送信する時		(mPIniFLAG)
		return

;==============================================================================;
;==		EP0o受信設定 (色々)
;==============================================================================;
;== EP0初期設定付き受信設定
USB_EP0_FirstSetup:
		movlw	b'00010110'				; ハンドシェイク,コントロール有効,OUT/INを
		movff	WREG, UEP0				; EP0に設定 (バンク対応)
;== EP0o 受信ストール設定 (セットアップパケットのみ可能)
USB_EP0oStall:
		movlw	fSIE|fBSTALL			; SIEオーナーとバッファストールを
		movwf	mBD0STAT				; mBD0STATに仮設定
;== BD0(EP0o)設定
USB_EP0oSetupBDT:
		movlw	low(BD0BUF)
		movwf	BD0ADRL					; データバッファの下位アドレス
		movlw	high(BD0BUF)
		movwf	BD0ADRH					; データバッファの下位アドレス
		movlw	EP0_BUF_SIZE
		movwf	BD0CNT					; 受信バッファサイズ
		movff	mBD0STAT, BD0STAT		; BD0STATに設定
		return

;== EP0o受信設定 (返答パケット用)
USB_EP0oDTS_All:
		movlw	fSIE
		movwf	mBD0STAT				; DTSに関係なく受信
		bra		USB_EP0oSetupBDT

;== EP0o受信設定 (DTS==1)
USB_EP0oDTS_1:
		bcf		BD0STAT, DTS			; DTS を強制的に0にする
;== EP0o受信設定 (DTS==反転)
USB_EP0oDTS_Toggle:
		movlw	fSIE|fDAT0|fDTSEN
		btfss	BD0STAT, DTS
		movlw	fSIE|fDAT1|fDTSEN
		movwf	mBD0STAT				; 前回のDTSを反転
		bra		USB_EP0oSetupBDT

;==============================================================================;
;==		EP0i(送信)各設定
;==============================================================================;
;== EP0i送信設定 (DTS==1)
USB_EP0iDTS_1:
		bcf		BD1STAT, DTS			; DTS を強制的に0にする
;== EP0i送信設定 (DTS==反転)
USB_EP0iDTS_Toggle:
		movlw	low(BD1BUF)
		movwf	BD1ADRL					; データバッファの下位アドレス
		movlw	high(BD1BUF)
		movwf	BD1ADRH					; データバッファの下位アドレス
		movff	mSNDCNT,  BD1CNT		; 送信バッファサイズ
		movlw	fSIE|fDAT0|fDTSEN
		btfss	BD1STAT, DTS
		movlw	fSIE|fDAT1|fDTSEN		; 前回のDTSを反転
		movwf	BD1STAT					; 送信
		return

;==============================================================================;
;==		EP0i送信ストール設定
;==============================================================================;
USB_EP0iStall:
		movlw	fSIE|fBSTALL			; SIEオーナーとバッファストールを
		movwf	BD1STAT					; 送信ストール
		return

;==============================================================================;
;==		EP1o(受信)各設定
;==============================================================================;
;== EP1o初期設定 (DTS==0)
USB_EP1oInit:
		bsf		BD2STAT, DTS			; DTS を強制的に1にする
;== EP1o受信設定 (DTS==反転)
USB_EP1oDTS_Toggle:
		movlw	low(BD2BUF)
		movwf	BD2ADRL					; データバッファの下位アドレス
		movlw	high(BD2BUF)
		movwf	BD2ADRH					; データバッファの下位アドレス
		movlw	EP1_BUF_SIZE
		movwf	BD2CNT					; 受信バッファサイズ
		movlw	fSIE|fDAT0|fDTSEN
		btfss	BD2STAT, DTS
		movlw	fSIE|fDAT1|fDTSEN		; 前回のDTSを反転
		movwf	BD2STAT					; 受信
		return

;==============================================================================;
;==		EP1i(送信)各設定
;==============================================================================;
;== EP1i初期設定 (DTS==0)
USB_EP1iInit:
		movlw	fCPU|fDAT1				; DTSを1にして
		movwf	BD3STAT					; 停止
		return
;== EP1i送信設定 (DTS==反転)
USB_EP1iDTS_Toggle:
		movlw	low(BD3BUF)
		movwf	BD3ADRL					; データバッファの下位アドレス
		movlw	high(BD3BUF)
		movwf	BD3ADRH					; データバッファの下位アドレス
		movff	mSNDCNT,  BD3CNT		; 送信バッファサイズ
		movlw	fSIE|fDAT0|fDTSEN
		btfss	BD3STAT, DTS
		movlw	fSIE|fDAT1|fDTSEN		; 前回のDTSを反転
		movwf	BD3STAT					; 送信
		return

;==============================================================================;
			END