/******************************************************************************
 * AC-DCアダプター用電圧/電流/電力計					Create Date 2017/03/11
 * File	Name	: vaw.c								Last Update 2017/03/21
 * Device		: PIC16F1827-I/P
 * Complier		: XC8 Ver 1.41
 * Author		: saka
 *
 * TAB = 4
 ******************************************************************************/
/* 電流計の試作回路ができたので一応プログラムを作りました。
 * フルスケール2A 0.01A刻み 精度2.5% 及び フルスケール200mA 1mA刻み 精度2.5%です。
 * 無調整で動作するようです。(floatで計算する必要もなかった。)
 * ADCが2mAステップでこの精度が出るならOKでしょう。
 * 後で高精度化のテストしてみるしかないかな？ その為に補正用の定電流回路も組んで
 * いるだから…
 */
/** Include Files *************************************************************/
#include <xc.h>
#include <stdint.h>
#include <stdbool.h>
#include "Disp.h"

// 使用クロック設定
#ifndef _XTAL_FREQ
	#define _XTAL_FREQ 16000000		// 16MHz
#endif

/** PIC16F1827 Configuration Bit Settings *************************************/
// CONFIG1
#pragma config FOSC		= INTOSC	// 内部発振器を有効する（16MHzに設定すること）
#pragma config WDTE		= ON		// ウォッチドックタイマを有効する
#pragma config PWRTE	= OFF		// パワーアップタイマを無効する
#pragma config MCLRE	= OFF		// MCLRを無効にする（入力ピンに使用）
#pragma config CP		= OFF		// プログラムメモリをプロテクトしない
#pragma config CPD		= OFF		// データメモリをプロテクトしない
#pragma config BOREN	= ON		// ブラウン・アウト・リセットを有効にする
#pragma config CLKOUTEN	= OFF		// クロック出力を無効にする
#pragma config IESO		= ON		// 内部/外部切り替えを有効にする
#pragma config FCMEN	= ON		// フェイル・セーフ クロック監視を有効にする
// CONFIG2
#pragma config WRT		= OFF		// フラッシュメモリの自己書き換えをプロテクトしない
#pragma config PLLEN	= ON		// 4x PLL を有効にする （使用しないけど）
#pragma config STVREN	= ON		// スタック・オーバーフローしたらリセットする
#pragma config BORV		= LO		// ブラウン・アウト・リセット電圧はLOにする
#pragma config LVP		= OFF		// 低電圧プログラミングは無効にする

/** Define ********************************************************************/
// 電圧入力の分圧抵抗比（10倍して整数に）
#define REGDIV	32					// ((1k+2.2k)/1k)*10 = 32 
// 各設定
#define VOLT_CH			4			// AN4 アナログ入力
#define CURRENT_CH		3			// AN3 アナログ入力
#define VOLT_UNIT		LED2		// 電圧 (V)
#define CURRENT_UNIT	LED3		// 電流 (A)
#define WATT_UNIT		LED4		// 電力 (W)
#define WATTHOUR_UNIT	LED5		// 電力量 (Wh)
// ADC値直接表示
//#define DISP_ADC					// ADCテスト
// (unsigned int) 1/10に最下位桁を四捨五入
#define mRoundOff(a) ((a%10<5)?a/10:a/10+1)

/** I/O Port ******************************************************************/
#define	InputSW		PORTAbits.RA5	// スイッチ入力

/** Variables *****************************************************************/
// 表示モードフラグ値
enum  _DISPMODE {
	DISP_VOLT = 0,					// 電圧表示
	DISP_CURRENT,					// 電流表示
	DISP_WATT,						// 電力表示
	DISP_WATTHUOR,					// 1時間当たりの電力量表示
	DISP_SETUP						// 調整モード
};
// メインフラグ
union _MAINFLAG {
	unsigned char  byte;
	struct {
		unsigned	f100mS	: 1;	// 100mSフラグ
		unsigned	f1S		: 1;	// 1秒フラグ
		unsigned	mode	: 2;	// 表示モードフラグ
		unsigned	sw		: 1;	// スイッチ
		unsigned	swp		: 1;	// スイッチ (前回)
		unsigned	swl		: 1;	// スイッチ (長期)
		unsigned			: 1;
	} bits;
};
static union _MAINFLAG mMFL;
static char mTMR2;

/** Internal Function *********************************************************/
void  interrupt Isr(void);
void DACOutput(char num);
uint16_t ADCInput(uint8_t ch);
uint16_t VoltData(void);
uint16_t CurrentData(void);
void DispVolt(uint16_t volt);
void DispCurrent(uint16_t current);
void DispWatt(uint16_t watt);
void DispWattHuor(uint16_t watthuor);
void Wait_100mS(uint8_t num);

/** Main Function *************************************************************/
// メイン処理
void main(void) {
	CLRWDT();
	OSCCON = 0x7A;					// 16MHz
	ANSELA = 0b00011100;			// ANSA4-ANSA2 Analog input Pin
	ANSELB = 0x00;					// Digital  I/O 
	// Fixed Voltage Reference
	FVRCON = 0b11000110;			// FVR ON / CDFVR:1.024V / ADFVR:2.048V
	// 定電流値出力
	DACOutput(0);					// DAC 出力 num * 0.032V (3.2mA)
	// 表示器初期化
	InitializeDisp();
	// Timer0 128uS 設定
	OPTION_REGbits.TMR0CS = 0;		// Fos/4
	OPTION_REGbits.TMR0SE = 0;		// low-to-high
	OPTION_REGbits.PSA    = 0;		// Prescaler
	OPTION_REGbits.PS     = 0b000;	// 1:2
	// Timer2 10mS 設定
	TMR2 = 0;						// Timer Clear
	PR2  = 250;						// 250 count
	T2CON = 0b01001110;				// 1:16 / Timer2 ON / 1:10
	// 割り込み設定
	INTCON = 0b01100000;			// PEIE:1 / TMR0IE:1 / TMR0IF:0
	PIR1bits.TMR2IF = 0;
	PIE1bits.TMR2IE = 1;
	INTCONbits.GIE  = 1;			// 割り込み開始
	// 各フラグクリア
	mMFL.byte = 0;
	// 表示テスト
	PostDisp();						// 全点灯試験
	Wait_100mS(10);
	ClearDisp();					// 消灯
	Wait_100mS(1);
	// 各データ初期化
	uint16_t nVolt;					// 電圧値
	uint16_t nVBuf[10];				// 電圧値の移動平均処理用 (1秒)
	uint16_t nCurrent;				// 電流値
	uint16_t nCBuf[10];				// 電流値の移動平均処理用 (1秒)
	uint16_t nWatt;					// 電力値 = 電圧値 × 電流値
	uint16_t nWh = 0;				// 電力量値 = 2分積算 × 30回 (1時間)の平均値
	uint16_t nWBuf[30];				// 2分積算を30回分(1時間分)保存する
	uint24_t slW2m = 0;				// 2分積算 1秒おきに2分間 電力値を積算値  
	int nCNT = 0;					// 移動平均処理用カウンタ
	int nCTs = 0;					// 2分積算用カウンタ
	int nCTm = 0;					// 1時間用カウンタ
	for (int i=0; i<10; i++) {		// 初期値代入
		nVBuf[i]	= VoltData();
		nCBuf[i]	= CurrentData();
	}
	for (int i=0; i<30; i++) nWBuf[i] = 0;
	// 5V時 7セグの輝度を落とす。
	if (nVBuf[0] > 4000) DispBright(5, 5);
	// メインループ
	while (1) {
		CLRWDT();
		if (mMFL.bits.f100mS) {
			mMFL.bits.f100mS = 0;
			// 各データ取得
			mMFL.bits.sw = InputSW;
			// 移動平均 電圧/電流
			nVBuf[nCNT]	= VoltData();
			nCBuf[nCNT]	= CurrentData();
			if (++nCNT == 10) nCNT = 0;
			nVolt = nCurrent = 0;
			for (int i=0; i<10; i++ ) {
				nVolt    += nVBuf[i];
				nCurrent += nCBuf[i]; 
			}
			nVolt    = mRoundOff( nVolt );
			nCurrent = mRoundOff( nCurrent ); 
			// 電力 (5000*2000で桁あふれする為)
			nWatt = (uint16_t)((long)nVolt * (long)nCurrent / 1000);
			// スイッチ処理（押した時）
			if (mMFL.bits.sw == 0 && mMFL.bits.swp == 1) {
				mMFL.bits.mode++;		// モード変更
			}
			mMFL.bits.swp = mMFL.bits.sw;
			// 各表示
			switch (mMFL.bits.mode) {
				case DISP_VOLT		: DispVolt( nVolt );		// 電圧
									  break;
				case DISP_CURRENT	: DispCurrent( nCurrent );	// 電流
									  break;
				case DISP_WATT		: DispWatt( nWatt );		// 電力
									  break;
				case DISP_WATTHUOR	: DispWattHuor( nWh );		// 電力量
									  break;
				default : break;
			}
			// 1秒処理
			if (!mMFL.bits.f1S) continue;
			mMFL.bits.f1S = 0;
			slW2m += (short long)nWatt;
			// 2分処理
			if (++nCTs < 120)	continue;
			nCTs = 0;
			nWBuf[nCTm] = (int)(slW2m / 120);
			slW2m = 0;
			// 電力量
			nWh = 0;
			for (int i=0; i<30; i++) nWh += nWBuf[i]; 
			nWh /= 30;
			// 1時間処理
			if (++nCTm < 30) 	continue;
			nCTm = 0;
		}
	}
}
// 割り込み処理
void  interrupt Isr(void)
{
	// Timer0 (7Seg Diplay)
	if (INTCONbits.TMR0IF == 1) {
		INTCONbits.TMR0IF = 0;
		InterruptDisp ();
	}
	// Timer2 (Main)
	if (PIR1bits.TMR2IF == 1) {
		PIR1bits.TMR2IF = 0;
		mTMR2++;
		if (mTMR2%10 == 0) {
			mMFL.bits.f100mS = 1;
		}
		if (mTMR2 == 100) {
			mTMR2 = 0;
			mMFL.bits.f1S = 1;
		}
	}
}
// Digital-to-Analog Converter
void DACOutput(char num)
{
	if (num > 32) {
		DACCON0 = 0b01101000;		// DAC OFF / Pos / DACOUT / FVR / VSS
		DACCON1 = 31;				// 31
	}
	else if (num <= 0) {
		DACCON0 = 0b00101000;		// DAC OFF / Neg / DACOUT / FVR / VSS
		DACCON1 = 0;				// 0
	} 
	else {
		DACCON0 = 0b11101000;		// DAC ON / DACOUT / FVR / VSS
		DACCON1 = num;				// num * 0.0032V
	}
}
// Analog-to-Digital Converter (Average16 / 3.2mS)
#define ADCCNT	16
uint16_t ADCInput(uint8_t ch)
{
	unsigned int adc = 0;
	ADCON1 = 0b11010011;			// Right / FOSC/16 / Vss / FVR(2.048V)
	ADCON0bits.CHS = ch;			// Analog Channel Select
	for (int i=0; i<ADCCNT; i++) {
		ADCON0bits.ADON = 1;		// ADC ON
		__delay_us(200);			// Acquisition Time (200uS)
		ADCON0bits.GO = 1;			// ADC Start
		while (ADCON0bits.GO_nDONE); 
		ADCON0bits.ADON = 0;		// ADC OFF
		adc += ADRES;
	}
	return adc / ADCCNT;
}
// 電圧値取得（固定小数点）
uint16_t VoltData(void)
{
#ifndef DISP_ADC
	// Rs1 と Rs2 抵抗分圧比 の10倍
	// Vref:2.048V / ADC:1024 = 0.002V * 1000倍
	// 固定小数点の位置で 1000倍 = (10倍 * 1000倍) / 10
	return (ADCInput(VOLT_CH) * 2 * REGDIV) / 10;
#else
	return ADCInput(VOLT_CH);
#endif
}
// 電流値取得（固定小数点）
uint16_t CurrentData(void)
{
#ifndef DISP_ADC
	//	2A = 2V の 1倍
	//	Vref:2.048V / ADC:1024 = 0.002V * 1000倍
	// 固定小数点の位置で 1000倍 = 1000倍 / 1;
	return (ADCInput(CURRENT_CH) * 2);
#else
	return ADCInput(CURRENT_CH);
#endif
}
// 電圧表示
void DispVolt(uint16_t volt)
{
#ifndef DISP_ADC
	DispFixedPoint( volt );
#else
	DispNumber( volt, false );
#endif
	DispUnit(VOLT_UNIT);
}
// 電流表示
void DispCurrent(uint16_t current)
{
#ifndef DISP_ADC
	DispFixedPoint( current );
#else
	DispNumber( current, false );
#endif
	DispUnit(CURRENT_UNIT);
}
// 電力表示
void DispWatt(uint16_t watt)
{
#ifndef DISP_ADC
	DispFixedPoint( watt );
#else
	DispNumber(0, false);
#endif
	DispUnit(WATT_UNIT);
}
// 電力量表示
void DispWattHuor(uint16_t watthuor)
{
#ifndef DISP_ADC
	DispFixedPoint( watthuor );
#else
	DispNumber(0, false);
#endif
	DispUnit(WATTHOUR_UNIT);
}
// 割り込み対応時間待ち
void Wait_100mS(uint8_t num)
{
	for (int i=0; i<num;) {
		CLRWDT();
		if (mMFL.bits.f100mS) {
			mMFL.bits.f100mS = 0;
			i++;
		}
	}
}
/**	end of file	***************************************************************/