/******************************************************************************
 * 7Seg 3column display for PIC16F					Create Date	2017/03/11
 * File Name	: disp.c							Last Update	2017/03/21
 * Complier		: XC8 Ver 1.41
 *
 * TAB size = 4
 ******************************************************************************/
/** Include Files *************************************************************/
#include <xc.h>
#include <stdint.h>
#include <stdbool.h>
#include "Disp.h"

/** Define ********************************************************************/
#define	OUTPIN	0
#define	INPIN	1
// (unsigned int) 1/10に最下位桁を四捨五入
#define mRoundOff(a) ((a%10<5)?a/10:a/10+1)
// 7Seg情報
#ifdef	COMMON_ANODE
	#define	COMMON	~
	#define	FT_Clr	0b11111111
	#define	FT_Set	0b00000000
	#define	ON	0
	#define	OFF	1
#else
	#define	COMMON
	#define	FT_Clr	0b00000000
	#define	FT_Set	0b11111111
	#define	ON	1
	#define	OFF	0
#endif
// 7Seg表示セグメント
#define	_A		0b00000001
#define	_B		0b00000010
#define	_C		0b00000100
#define	_D		0b00001000
#define	_E		0b00010000
#define	_F		0b00100000
#define	_G		0b01000000
#define	D_P		0b10000000
// 7Seg表示字形
#define	FT0	COMMON(_A+_B+_C+_D+_E+_F)
#define	FT1	COMMON(_B+_C)
#define	FT2	COMMON(_A+_B+_D+_E+_G)
#define	FT3	COMMON(_A+_B+_C+_D+_G)
#define	FT4	COMMON(_B+_C+_F+_G)
#define	FT5	COMMON(_A+_C+_D+_F+_G)
#define	FT6	COMMON(_A+_C+_D+_E+_F+_G)
#define	FT7	COMMON(_A+_B+_C)
#define	FT8	COMMON(_A+_B+_C+_D+_E+_F+_G)
#define	FT9	COMMON(_A+_B+_C+_D+_F+_G)
#define	FTA	COMMON(_A+_B+_C+_E+_F+_G)
#define	FTb	COMMON(_C+_D+_E+_F+_G)
#define	FTC	COMMON(_A+_D+_E+_F)
#define	FTd	COMMON(_B+_C+_D+_E+_G)
#define	FTE	COMMON(_A+_D+_E+_F+_G)
#define	FTF	COMMON(_A+_E+_F+_G)
#define	FTP	COMMON(_A+_B+_E+_F+_G)
#define	ERH	COMMON(_A)
#define	ERL	COMMON(_D)

/** Variables *****************************************************************/
// 割り込み表示のステート
unsigned char nDispState;
unsigned char nDPxOFF = 0x0E;
unsigned char nLedOFF = 0x0E;
enum DISPSTATE {
	DP1_ON   = 0x00,
	DP2_ON   = 0x10,
	DP3_ON   = 0x20,
	LED_ON   = 0x30,
	DISP_END = 0x40
};
// 7Segの表示データ
static unsigned char NumFont[] =
	{ FT0, FT1, FT2, FT3, FT4, FT5, FT6, FT7, FT8, FT9,
	  FTA, FTb, FTC, FTd, FTE, FTF };

// 7Seg表示セグメント(上記"7Seg表示セグメント"と位置を合わせる事)
union _7SEG {
	unsigned char  byte;
	struct {
		unsigned	A		: 1;	// A segment
		unsigned	B		: 1;	// B segment
		unsigned	C		: 1;	// C segment
		unsigned	D		: 1;	// D segment
		unsigned	E		: 1;	// E segment
		unsigned	F		: 1;	// F segment
		unsigned	G		: 1;	// G segment
		unsigned	DP		: 1;	// Dot Point
	} bits;
};
union _7SEG nDisp1, nDisp2, nDisp3;
// LED表示
union _LEDs {
	unsigned char  byte;
	struct {
		unsigned	LD1	: 1;	// LED1 (V)
		unsigned	LD2	: 1;	// LED2 (A)
		unsigned	LD3	: 1;	// LED3 (W)
		unsigned	LD4	: 1;	// LED4 (Wh)
		unsigned	LD5	: 1;	// LED5 (Option)
		unsigned		: 3;
	} bits;
};
union _LEDs nLeds;

/** Function ******************************************************************/
// 表示器の初期化
void InitializeDisp(void)
{
	// COMMON
	DP1		= OFF;
	DP1TR	= OUTPIN;
	DP2		= OFF;
	DP2TR	= OUTPIN;
	DP3		= OFF;
	DP3TR	= OUTPIN;
	LEDs	= OFF;
	LEDsTR	= OUTPIN;
	// SEGMENT
	SEGs = FT_Clr;
	SEGsTR = 0x00;		// OUTPUT
	// Datas
	nDispState = 0;
}
// 電源時試験表示 
void PostDisp(void)
{
	nDisp1.byte = FT_Set;
	nDisp2.byte = FT_Set;
	nDisp3.byte = FT_Set;
	nLeds.byte  = FT_Set;
}
// 非表示
void ClearDisp(void)
{
	nDisp1.byte = FT_Clr;
	nDisp2.byte = FT_Clr;
	nDisp3.byte = FT_Clr;
	nLeds.byte  = FT_Clr;
}
// 割り込み表示処理部（128uS割り込み）
// 表示を切り替える時は必ず消してから次に移ること。
void InterruptDisp (void)
{
	nDispState++;
	if (nDispState >= DISP_END) nDispState= 0x00;
	if ((nDispState & 0x0F) == nDPxOFF){
		DP1 = OFF;
		DP2 = OFF;
		DP3 = OFF;
	}
	if ((nDispState & 0x0F) == nLedOFF){
		LEDs = OFF;
	}
	switch (nDispState) {
		case DP1_ON :
			SEGs = nDisp1.byte;
			DP1 = ON;
			break;
		case DP2_ON :
			SEGs = nDisp2.byte;
			DP2 = ON;
			break;
		case DP3_ON :
			SEGs = nDisp3.byte;
			DP3 = ON;
			break;
		case LED_ON :
			SEGs = nLeds.byte;
			LEDs = ON;
			break;
		default:
			break;
	}
}
// 表示器の輝度変更
void DispBright(uint8_t dp, uint8_t led)
{
	if (dp > 0x0F)	nDPxOFF = 0x0E;
	else			nDPxOFF = dp;
	if (led > 0x0F)	nLedOFF = 0x0E;
	else			nLedOFF = led;
}
// 数字表示  000～999
// xc8 ver1.41で int の %（余り）を計算すると
// advisory: (1510) non-reentrant function "___awmod" appears in multiple call
// graphs and has been duplicated by the compiler
// の メッセージが出る。unsigned int は出ない。
// マイナス側は、使わないので問題なし
bool DispNumber(int16_t num, bool blank)
{
	if (num < 0) {
		nDisp3.byte = ERH;
		nDisp2.byte = ERH;
		nDisp1.byte = ERH;
		return false;
	}
	else if (num > 1000) {
		nDisp3.byte = ERH;
		nDisp2.byte = ERH;
		nDisp1.byte = ERH;
		return false;
	}
	// 数字表示
	nDisp3.byte = NumFont[(uint16_t)num % 10];
	num = num / 10;
	if (blank && num == 0)	nDisp2.byte = FT_Clr;
	else					nDisp2.byte = NumFont[(uint16_t)num % 10];
	num = num / 10;
	if (blank && num == 0)	nDisp1.byte = FT_Clr;
	else					nDisp1.byte = NumFont[(uint16_t)num % 10];
	return true;
}
// 固定小数点表示（下位3桁が小数点以下の数値）
// 最大99.944 まで表示
bool DispFixedPoint(uint16_t num)
{
	char dot;
	// 表示用数値変更部
	if (num < 200) {
		nLeds.bits.LD1 = ON;
		return DispNumber(num, true);
	}
	nLeds.bits.LD1 = OFF;
	num = mRoundOff( num );
	if (num <= 999) {
		dot = 1;
	}
	else if (num <= 9994) {
		dot = 2;
		num = mRoundOff( num );
	}
	else {
		nDisp3.byte = ERH;
		nDisp2.byte = ERH;
		nDisp1.byte = ERH;
		return false;
	}
	// 数字表示
	DispNumber(num, false);
	// 小数点表示
	if		(dot == 1)	nDisp1.bits.DP = ON;
	else if	(dot == 2)	nDisp2.bits.DP = ON;
	return true;
}
// LED(単位)表示部
void DispUnit(uint8_t num)
{
	nLeds.bits.LD2 = OFF;
	nLeds.bits.LD3 = OFF;
	nLeds.bits.LD4 = OFF;
	nLeds.bits.LD5 = OFF;
	switch (num) {
		case 1: nLeds.bits.LD2 = ON;
				break;
		case 2: nLeds.bits.LD3 = ON;
				break;
		case 3: nLeds.bits.LD4 = ON;
				break;
		case 4: nLeds.bits.LD5 = ON;
				break;
		default:
				break;
	}
}
/**	end of file	***************************************************************/