/****************************************************************************** 
 ** PIC USB HID Device								Create Date	2015/11/08
 **			for PIC18 (Non-J Family)				Last Update	2016/03/12
 ** file name :  usbhiddev.c								Saka Softwares
 ** language  :  XC8 (v1.36)
 ******************************************************************************
 ** XC8言語(専用)でUSB HID デバイスを作成できます。
 ** USBの処理は、割り込み専用です。
 ** Microchip製のusb_device.cを基にして記述していますが構造はかなり違います。
 ** 変更する場合は気を付けてください。
 ******************************************************************************
 ** 制限事項
 ** コンフィグレーションディスクリプタは、1つしか扱えません。
 ** インターフェースディスクリプタも基本的には1つしか持てません。
 ** (複数対応には USBCheckHIDRequest() の修正が必要です。)
 ** エンドポイントは、IN/OUTを１組として最大で７組が使用できます。
 ** Get Status等一部テストしていない機能が有ります。
 ******************************************************************************/
/** Version info. *************************************************************
 **	1.3		2016/03/12	送信処理の後始末の不具合/その他を修正
 **						ワーニング抑制を追加 初期値 -3 でコンパイルするように変更
 **	1.2		2016/01/20	バグ修正 (パイプ フラグに間違い等)
 **	1.1		2015/12/07	HID Keyboard Demo 対応のため関数追加
 **	1.0		2015/11/19	公開バージョン
 ******************************************************************************/
/** Include Files *************************************************************/
#include <xc.h>
#include <stdint.h>
#include <stdbool.h>
/** Warning Disable ***********************************************************/
// 下記のワーニングを抑制しています。エラーレベル -3 でコンパイルできます。
#pragma warning disable 520		//warning: (520) function "_XXX" is never called
#pragma warning disable 751		//warning: (751) arithmetic overflow in constant expression
#pragma warning disable 752		//warning: (752) conversion to shorter data type
#pragma warning disable 759		//warning: (759) expression generates no code
/** Include Files *************************************************************/
#include "usbdesc.h"
#include "usbhiddev.h"

/** Location fixing for USB Variable ******************************************/
// BDTの使用数 (ピンポンバッファは使用しない)
#define BDT_NUM_ENTRIES	(USB_USE_ENDPOINTS * 2)

// BD用バッファアドレス (ピンポンバッファを使用しない)
#define	BD0BUF_ADDRESS	(BDT_NUM_ENTRIES * 4 + USB_RAM__ADDR)
#define	BD1BUF_ADDRESS	(BD0BUF_ADDRESS  + EP0_BUFF_SIZE)
#define	BD2BUF_ADDRESS	(BD1BUF_ADDRESS  + EP0_BUFF_SIZE)
#define	BD3BUF_ADDRESS	(BD2BUF_ADDRESS  + EP1_OUT_BUFF_SIZE)
#define	BD4BUF_ADDRESS	(BD3BUF_ADDRESS  + EP1_IN_BUFF_SIZE)
#define	BD5BUF_ADDRESS	(BD4BUF_ADDRESS  + EP2_OUT_BUFF_SIZE)
#define	BD6BUF_ADDRESS	(BD5BUF_ADDRESS  + EP2_IN_BUFF_SIZE)
#define	BD7BUF_ADDRESS	(BD6BUF_ADDRESS  + EP3_OUT_BUFF_SIZE)
#define	BD8BUF_ADDRESS	(BD7BUF_ADDRESS  + EP3_IN_BUFF_SIZE)
#define	BD9BUF_ADDRESS	(BD8BUF_ADDRESS  + EP4_OUT_BUFF_SIZE)
#define	BD10BUF_ADDRESS	(BD9BUF_ADDRESS  + EP4_IN_BUFF_SIZE)
#define	BD11BUF_ADDRESS	(BD10BUF_ADDRESS + EP5_OUT_BUFF_SIZE)
#define	BD12BUF_ADDRESS	(BD11BUF_ADDRESS + EP5_IN_BUFF_SIZE)
#define	BD13BUF_ADDRESS	(BD12BUF_ADDRESS + EP6_OUT_BUFF_SIZE)
#define	BD14BUF_ADDRESS	(BD13BUF_ADDRESS + EP6_IN_BUFF_SIZE)
#define	BD15BUF_ADDRESS	(BD14BUF_ADDRESS + EP7_OUT_BUFF_SIZE)

//  バッファ ディスクリプタ テーブル
volatile BDT_ENTRY	BD[BDT_NUM_ENTRIES]			@USB_RAM__ADDR;
#define ep0Bo	BD[0]
#define ep0Bi	BD[1]

// エンドポイントIN/OUT バッファ (ピンポンバッファを使用しない)
volatile CTRL_TRF_SETUP	SetupPkt				@BD0BUF_ADDRESS;
volatile CTRL_TRF_DATA	DataPkt					@BD1BUF_ADDRESS;
#if(1 < USB_USE_ENDPOINTS)
volatile uint8_t ep1Bobuf[ EP1_OUT_BUFF_SIZE ]	@BD2BUF_ADDRESS;
volatile uint8_t ep1Bibuf[ EP1_IN_BUFF_SIZE ]	@BD3BUF_ADDRESS;
#endif
#if(2 < USB_USE_ENDPOINTS)
volatile uint8_t ep2Bobuf[ EP2_OUT_BUFF_SIZE ]	@BD4BUF_ADDRESS;
volatile uint8_t ep2Bibuf[ EP2_IN_BUFF_SIZE ]	@BD5BUF_ADDRESS;
#endif
#if(3 < USB_USE_ENDPOINTS)
volatile uint8_t ep3Bobuf[ EP3_OUT_BUFF_SIZE ]	@BD6BUF_ADDRESS;
volatile uint8_t ep3Bibuf[ EP3_IN_BUFF_SIZE ]	@BD7BUF_ADDRESS;
#endif
#if(4 < USB_USE_ENDPOINTS)
volatile uint8_t ep4Bobuf[ EP4_OUT_BUFF_SIZE ]	@BD8BUF_ADDRESS;
volatile uint8_t ep4Bibuf[ EP4_IN_BUFF_SIZE ]	@BD9BUF_ADDRESS;
#endif
#if(5 < USB_USE_ENDPOINTS)
volatile uint8_t ep5Bobuf[ EP5_OUT_BUFF_SIZE ]	@BD10BUF_ADDRESS;
volatile uint8_t ep5Bibuf[ EP5_IN_BUFF_SIZE ]	@BD11BUF_ADDRESS;
#endif
#if(6 < USB_USE_ENDPOINTS)
volatile uint8_t ep6Bobuf[ EP6_OUT_BUFF_SIZE ]	@BD12BUF_ADDRESS;
volatile uint8_t ep6Bibuf[ EP6_IN_BUFF_SIZE ]	@BD13BUF_ADDRESS;
#endif
#if(7 < USB_USE_ENDPOINTS)
volatile uint8_t ep7Bobuf[ EP7_OUT_BUFF_SIZE ]	@BD14BUF_ADDRESS;
volatile uint8_t ep7Bibuf[ EP7_IN_BUFF_SIZE ]	@BD15BUF_ADDRESS;
#endif

/** Fixed number for USB ******************************************************/
// EPnOut(受信)バッファサイズ
const uint8_t EPnOutBufSize[] = {
	EP0_BUFF_SIZE,
#if(1 < USB_USE_ENDPOINTS)
	EP1_OUT_BUFF_SIZE,
#endif
#if(2 < USB_USE_ENDPOINTS)
	EP2_OUT_BUFF_SIZE,
#endif
#if(3 < USB_USE_ENDPOINTS)
	EP3_OUT_BUFF_SIZE,
#endif
#if(4 < USB_USE_ENDPOINTS)
	EP4_OUT_BUFF_SIZE,
#endif
#if(5 < USB_USE_ENDPOINTS)
	EP5_OUT_BUFF_SIZE,
#endif
#if(6 < USB_USE_ENDPOINTS)
	EP6_OUT_BUFF_SIZE,
#endif
#if(7 < USB_USE_ENDPOINTS)
	EP7_OUT_BUFF_SIZE,
#endif
};

// EPnIn(送信)バッファサイズ
const uint8_t EPnInBufSize[] = {
	EP0_BUFF_SIZE,
#if(1 < USB_USE_ENDPOINTS)
	EP1_IN_BUFF_SIZE,
#endif
#if(2 < USB_USE_ENDPOINTS)
	EP2_IN_BUFF_SIZE,
#endif
#if(3 < USB_USE_ENDPOINTS)
	EP3_IN_BUFF_SIZE,
#endif
#if(4 < USB_USE_ENDPOINTS)
	EP4_IN_BUFF_SIZE,
#endif
#if(5 < USB_USE_ENDPOINTS)
	EP5_IN_BUFF_SIZE,
#endif
#if(6 < USB_USE_ENDPOINTS)
	EP6_IN_BUFF_SIZE,
#endif
#if(7 < USB_USE_ENDPOINTS)
	EP7_IN_BUFF_SIZE,
#endif
};

// 各BDT(EPnOut/In)用バッファのアドレス一覧
const uint16_t BDnBufAdr[] = {
	BD0BUF_ADDRESS,			// SetupPkt
	BD1BUF_ADDRESS,			// DataPkt
#if(1 < USB_USE_ENDPOINTS)
	BD2BUF_ADDRESS,			// ep1Bobuf
	BD3BUF_ADDRESS,			// ep1Bibuf
#endif
#if(2 < USB_USE_ENDPOINTS)
	BD4BUF_ADDRESS,			// ep2Bobuf
	BD5BUF_ADDRESS,			// EP2iBUF
#endif
#if(3 < USB_USE_ENDPOINTS)
	BD6BUF_ADDRESS,			// EP3oBUF
	BD7BUF_ADDRESS,			// EP3iBUF
#endif
#if(4 < USB_USE_ENDPOINTS)
	BD8BUF_ADDRESS,			// EP4oBUF
	BD90BUF_ADDRESS,		// EP4iBUF
#endif
#if(5 < USB_USE_ENDPOINTS)
	BD10BUF_ADDRESS,		// EP5oBUF
	BD11BUF_ADDRESS,		// EP5iBUF
#endif
#if(6 < USB_USE_ENDPOINTS)
	BD12BUF_ADDRESS,		// EP6oBUF
	BD13BUF_ADDRESS,		// EP6iBUF
#endif
#if(7 < USB_USE_ENDPOINTS)
	BD14BUF_ADDRESS,		// EP7oBUF
	BD15BUF_ADDRESS,		// EP7iBUF
#endif
};
// EPnからBDnへ番号変換
#define ep_BoNum(a) (a * 2)							// EPnOutからBDnへ
#define ep_BiNum(a) ((a * 2) + 1)					// EPnInからBDnへ

/** Variable ******************************************************************/
volatile CTRL_TRF_STAT controlTransferState;		// コントロール転送遷移

volatile uint8_t usb_active_cfg;					// USBコンフィグ値
volatile uint8_t usb_alt_intf[USB_MAX_DSC_INTF];	// USBインターフェース代替値
volatile bool    usb_remote_wakeup;					// USBリモートウェイクアップ
volatile uint8_t usb_idle_rate;						// USBアイドルレート
volatile uint8_t usb_active_protocol;				// USBプロトコル

volatile IN_PIPE ep0Pi;								// EP0送信用パイプ

volatile uint8_t UIEsave;							// USB 割り込み保存用

/** Internal Function *********************************************************/
void USBCheckStdRequest(void);
void USB_EP0Setup(void);
void USB_EP0Output(void);
void USB_EP0Input(void);
void setEP0oBD(uint8_t status);
void setEP0iBD(uint8_t status);
void USBCheckHIDRequest(void);

/******************************************************************************
 ** USB 初期化処理
 ** 引  数	: 無し
 ** 戻り値	: 無し
 ******************************************************************************/
void USBDeviceInit(void)
{
	// バスパワー検出設定
	#if defined(USE_USB_BUS_SENSE_IO)
		tris_usb_bus_sense();
	#else
		#define usb_bus_sense 1
	#endif
	// セルフパワー検出設定
	#if defined(USE_SELF_POWER_SENSE_IO)
		tris_self_power();
	#else
		#define self_power 0
    #endif
	USBIE_BIT = 1;							// 周辺割り込み USBIE 許可
	usb_idle_rate = 500 / 4;				// アイドルレート (500mS) (Ver1.1 追加)
	usb_active_protocol = 0;				// プロトコル (Boot)(Ver1.1 追加)
	USBDeviceState = DETACHED_STATE;
}

/******************************************************************************
 ** USBデバイスタスク
 ** 引  数	: 無し
 ** 戻り値	: USBDeviceState
 ******************************************************************************/
USB_DEVICE_STATE USBDeviceTasks(void)
{
	//*** バス接続処理処理 *****************************************************
	if (usb_bus_sense == 1) {
		if ( USBDeviceState == DETACHED_STATE ) {
			UIE  = 0x11;						// IDLEIE と URSTIEを許可
			UEIE = 0x00;						// USB エラー割り込み禁止
			UIR  = 0x00;						// USB フラグクリア
		    UCON = 0x00;						// USB 停止
		    UADDR = 0;							// デバイスアドレス 0 の設定
			UCFG = _PUEN | _FS;					// UCFG 設定（FS/Pull Up）
			while (!UCONbits.USBEN) {
				UCONbits.USBEN = 1;				// 確実にUSBモジュールを起動する
			}
			#ifdef ENSBLE_SOF_COUNTER
				USBSOFCounter = 0;				// SOF カウンタのクリア
			#endif
			USBIF_FLG = 0;						// 周辺割り込み USBIFをクリア
			USBDeviceState = ATTACHED_STATE;
		}
	}
	//*** バス切断処理処理 *****************************************************
	else {
		if ( USBDeviceState != DETACHED_STATE ) {
			UIE  = 0x00;						// USB 割り込み禁止
			UEIE = 0x00;						// USB エラー割り込み禁止
			UIR  = 0x00;						// USB フラグクリア
		    UCON = 0x00;						// USB 停止
			USBIF_FLG = 0;						// 周辺割り込み USBIFをクリア
			USBDeviceState = DETACHED_STATE;
		}
		return DETACHED_STATE;					// USBバスが未接続
	}
	return USBDeviceState;
}

/******************************************************************************
 ** USB割り込みタスク
 ** 引  数	: 無し
 ** 戻り値	: USBDeviceState
 ******************************************************************************/
void USBInterruptTasks(void)
{
	if (!USBIF_FLG) return;						// USB 周辺割り込み以外
	//** バスアクティビリティ処理 ***********************************************
	if(UIRbits.ACTVIF) {
		UCONbits.SUSPND = 0; 
		UIE = UIEsave;							// 割り込み設定を復元
		while (UIRbits.ACTVIF) {
			UIRbits.ACTVIF = 0;					// 確実にフラグをクリアする
		}
	}
	//** バスリセット処理 ******************************************************
	if (UIRbits.URSTIF) {
	    UADDR = 0;								// デバイスアドレス 0 の設定
	    usb_active_cfg = 0;						// USB configurationの初期化
		uint8_t i = BDT_NUM_ENTRIES;
		while (i--) {BD[i].Val = 0;}			// BDTの初期化
		i = USB_MAX_DSC_INTF;
		while (i--) {usb_alt_intf[i] = 0;}		// IFの代替値を初期化
		usb_remote_wakeup = 0;					// リモートウェイクアップを初期化
		usb_idle_rate = 500 / 4;				// 500mS にセット (キーボード)
		usb_active_protocol = 0;				// Boot Protocolにセット (キーボード／マウス)
		for (i = 1; i< USB_USE_ENDPOINTS; i++) {
			*(&UEP0 + i) = 0x00;				// 使用するEPを初期化する
		}
	    UEP0 = _CTRL | _HSHK;					// EP0の初期化
		setEP0oBD( _DAT0 | _BSTALL );			// 最初の受信の設定
		UIR = 0;								// 全てのフラグをクリア
		UIE  = 0x39;							// SOFIE,STALLIE,IDLEIE,TRNIE と URSTIEを許可
	    USBDeviceState = DEFAULT_STATE;
	}
	//** Start of Frame処理 ***************************************************
	if (UIRbits.SOFIF) {
		UIRbits.SOFIF = 0;
		#ifdef ENSBLE_SOF_COUNTER
		USBSOFCounter++;						// SOF が来るたびにカウントする
		#endif
	}
	//** バスアイドル処理 ******************************************************
	if (UIRbits.IDLEIF) {
		UIRbits.IDLEIF = 0;
		UCONbits.SUSPND = 1;					// サスペンドに移行
		UIEsave = UIE;							// 割り込み設定を保存
		UIE = 0x04;								// バス アクティビティのみ許可
	}
	//** バスストール処理 ******************************************************
	if (UIRbits.STALLIF) {
		if (UEP0bits.EPSTALL) {
			if (ep0Bo.STAT.UOWN &&
				ep0Bi.STAT.UOWN &&  ep0Bi.STAT.BSTALL) {
				setEP0oBD( _BSTALL );
			}
			UEP0bits.EPSTALL = 0;
		}
		UIRbits.STALLIF = 0;
	}
	/**** 通信処理 *************************************************************
	 * ここではEP0のコントロール通信のみを処理します。
	 * EP0以外は転送完了フラグをクリアするだけです。
	 **************************************************************************/
	uint8_t i = 4;
	while (i--) {
		if (!UIRbits.TRNIF)	break;				// 転送終了がない時
		else {
			USTATCOPY USTATcpy;					// フラグをクリアする前に
			USTATcpy.Val = USTAT;				// USTATレジスタを保存する
			UIRbits.TRNIF = 0;					// TRNIFフラグをクリアする。
			if (USTATcpy.ENDP == 0) {			// EP0 の転送か？
				if (USTATcpy.DIR == DIR_OUT) {
					if (ep0Bo.STAT.PID == SETUP_TOKEN) {
						USB_EP0Setup();			// セットアップ受信処理
					}
					else {
						USB_EP0Output();		// コントロール受信処理
					}
				}
				else {
					USB_EP0Input();				// コントロール送信処理
				}
			}
		}
	}
	USBIF_FLG = 0;								// 周辺割り込み USBIFをクリア
}

/******************************************************************************
 ** 受信中フラグ
 ** 引  数	: ep エンドポイント番号
 ** 戻り値	: true = 受信中 / false = 受信終了 (受信データ処理可)
 ******************************************************************************/
bool EPnOutIsBusy(uint8_t ep)
{
	return BD[ep_BoNum(ep)].STAT.UOWN;
}

/******************************************************************************
 ** 受信データ処理
 ** 引  数	: ep   エンドポイント番号
 **			: pbuf 受信バッファポインタ
 **			: len  受信バッファ数
 ** 戻り値	: 受信データ数
 ******************************************************************************/
uint8_t EPnOutReport(uint8_t ep, uint8_t* pbuf, uint8_t len)
{
	if (USBDeviceState != CONFIGURED_STATE) return RCV_ERROR;
	uint8_t bd = ep_BoNum(ep);
	if (!BD[bd].STAT.UOWN) {
		uint8_t ret = 0;
 		if (len > BD[bd].CNT) len = BD[bd].CNT;
		ret = len;
		while (len--) *pbuf++ = *BD[bd].ADR++;		// バッファ転送
		BD[bd].CNT = EPnOutBufSize[ep];
		BD[bd].ADR = (uint8_t*)BDnBufAdr[bd];
		// DTSトグル処理
		if (BD[bd].STAT.DTS)	BD[bd].STAT.Val = _DAT0 | _DTSEN;	
		else					BD[bd].STAT.Val = _DAT1 | _DTSEN;
		BD[bd].STAT.UOWN = 1;						// 受信
		return ret;
	}
	return RCV_RECEING;
}

/******************************************************************************
 ** 送信中フラグ
 ** 引  数	: ep エンドポイント番号
 ** 戻り値	: true = 送信中 / false = 送信終了(データ送信処理可) 
 ******************************************************************************/
bool EPnInIsBusy(uint8_t ep)
{
	return BD[ep_BiNum(ep)].STAT.UOWN;
}

/******************************************************************************
 ** データ送信処理 (修正 2016/03/02)
 ** 引  数	: ep   エンドポイント番号
 **			: pram RAMバッファポインタ
 **			: prom ROMバッファポインタ
 **			: len  送信データ数
 **			: standby
 ** 戻り値	: 無し
 ** pram，prom は、使わない側を0にしてください。
 ** standby をtrueにすると、送信の空きを待たずに戻ります。
 ******************************************************************************/
uint8_t EPnInReport(uint8_t ep, uint8_t* pram, const uint8_t* prom, uint8_t len, bool standby)
{
	if (USBDeviceState != CONFIGURED_STATE || (prom != 0 && pram != 0) ) {
		return SND_ERROR;
	}
	uint8_t bd = ep_BiNum(ep);
	if (!standby || !BD[bd].STAT.UOWN) {
		while (BD[bd].STAT.UOWN);
		uint8_t rst = 0;
		BD[bd].CNT = EPnInBufSize[ep];
		if (len > BD[bd].CNT)	len = BD[bd].CNT;
		else					rst = BD[bd].CNT - len;
		uint8_t ret = len;
		BD[bd].ADR = (uint8_t*)BDnBufAdr[bd];
		uint8_t* pDest = BD[bd].ADR;
		while (len--) {
			if (prom != 0)	*pDest++ = *prom++;		// ROMバッファ転送
			if (pram != 0)	*pDest++ = *pram++;		// RAMバッファ転送
		}
		while (rst--)		*pDest++ = 0;			// 残りを0に
		// DTSトグル処理
		if (BD[bd].STAT.DTS)	BD[bd].STAT.Val = _DAT0 | _DTSEN;	
		else					BD[bd].STAT.Val = _DAT1 | _DTSEN;
		BD[bd].STAT.UOWN = 1;						// 送信
		return ret;
	}
	return SND_STANDBY;
}

/******************************************************************************
 ** SOF(Start of Frame)のカウント数取得	(Ver1.1 追加)
 ** 引  数	: 無し
 ** 戻り値	: カウント数 (32bit) 
 ******************************************************************************/
uint32_t USBGetSOFCounter(void)
{
	uint32_t cnt = 0;
	#ifdef ENSBLE_SOF_COUNTER
	if (USBIE_BIT == 1) {		// 通信中か?
		USBIE_BIT = 0;			// 周辺割り込み USBIE 禁止
		cnt = USBSOFCounter;
		USBIE_BIT = 1;			// 周辺割り込み USBIE 許可
	}
	#endif
	return cnt;
}

/******************************************************************************
 ** アイドル時間の取得	(Ver1.1 追加)
 ** 引  数	: 無し
 ** 戻り値	: アイドル時間 (mS単位)
 **			  設定されているアイドルレートから時間に換算する。 
 ******************************************************************************/
uint32_t USBGetIdleTime(void)
{
	// 1 idlerateは 4mS (Max 1,020mS)
	return usb_idle_rate * 4;
}

/******************************************************************************
 ** 使用中のプロトコルの取得	(Ver1.1 追加)
 ** 引  数	: 無し
 ** 戻り値	: プロトコル (0:Boot 1:Repote) 
 ******************************************************************************/
uint8_t USBGetActiveProtocol(void)
{
	// 0:Boot 1:Report
	return usb_active_protocol;
}

/******************************************************************************
 ** 内部関数
 ******************************************************************************
 ** セットアップ受信処理
 ******************************************************************************/
void USB_EP0Setup(void)
{
	// セットアップ受信
    controlTransferState = WAIT_SETUP;
	// EP0iの制御用BDTを強制的にCPUオーナーにする	(無いと通信エラーになる)
	// EP0iのBDTをCPUオーナーにする
	ep0Bo.STAT.UOWN = 0;
	ep0Bi.STAT.UOWN = 0;
	// 通常の通信を止める為パイプをフラッシュする
    ep0Pi.flag.Val   = 0;
    ep0Pi.wCount.Val = 0;

	//	各セットアップ受信処理
    USBCheckStdRequest();								// スタンダートリクエス
	USBCheckHIDRequest();								// HIDクラスリクエスト
    UCONbits.PKTDIS = 0;								// パケット転送の許可

	// セットアップ送信処理
	if ( ep0Pi.flag.BUSY ) {
		if (SetupPkt.bmRequestType.DataDir == DEV_TO_HOST) {
			controlTransferState = CTRL_TRF_TX;
			// 要求された送信数を検査する
			if (SetupPkt.wLength.Val < ep0Pi.wCount.Val) {
				// 送信数を変更する
				ep0Pi.wCount.Val = SetupPkt.wLength.Val;
			}
			setEP0oBD( _DTSEN );					//返答受信バケット待ち
			setEP0iBD(_DAT1 | _DTSEN);				//最初の送信バケット
		}
		else {
			controlTransferState = CTRL_TRF_RX;
			if (SetupPkt.wLength.Val == 0) {
				setEP0iBD(_DAT1 | _DTSEN);			// 返答送信パケット
			}
			setEP0oBD( _BSTALL );					// セットアップバケット待ち
		}
	}
	else {
		setEP0iBD(_BSTALL);							// 送信中止
		setEP0oBD(_BSTALL);							// セットアップバケット待ち
	}
}

/******************************************************************************
 ** EP0 コントロール受信処理
 ******************************************************************************/
void USB_EP0Output(void)
{
	// コントロール送信の返答待ちか？
	if (controlTransferState == CTRL_TRF_TX) {
		setEP0oBD( _BSTALL );							// セットアップパケット待ち		
		controlTransferState = WAIT_SETUP;
	}
}

/******************************************************************************
 ** コントロール送信処理 (修正 2016/03/01)
 ******************************************************************************/
void USB_EP0Input(void)
{
	// アドレスペンディング処理
	if (USBDeviceState == ADR_PENDING_STATE) {
		UADDR = SetupPkt.wValue.SADR.bDevAdr;			// デバイスアドレスの設定
		if(UADDR > 0)	USBDeviceState = ADDRESS_STATE;	// アドレス設定
		else			USBDeviceState = DEFAULT_STATE;	// 初期へ戻る
	}
	// EP0 コントロール送信処理
    if ( controlTransferState == CTRL_TRF_TX ) {
		// 送信データがあるか？
		if ( ep0Pi.flag.BUSY ) {
			// 全ての転送が終わったか？ (2016/03/01 移動/追加)
			if (ep0Pi.wCount.Val == 0) {
				ep0Pi.flag.BUSY = 0;
				setEP0iBD(_BSTALL);						// 追加
			}
			else {
				// DTS 反転 送信処理
				if (ep0Bi.STAT.DTS)	setEP0iBD( _DAT0 | _DTSEN);
				else				setEP0iBD( _DAT1 | _DTSEN);
			}
        }
    }
}

/******************************************************************************
 ** コントロール受信開始設定
 ******************************************************************************/
void setEP0oBD(uint8_t status)
{
	ep0Bo.CNT		= EP0_BUFF_SIZE;
	ep0Bo.ADR		= (uint8_t*)&SetupPkt;
	ep0Bo.STAT.Val	= status;
	ep0Bo.STAT.UOWN	= 1;
}

/******************************************************************************
 ** コントロール送信開始設定 (修正 2016/03/01)
 ******************************************************************************/
void setEP0iBD(uint8_t status)
{
	ep0Bi.CNT = 0;
	if ( !ep0Pi.flag.ACK ) {
		// 送信する転送数
		uint8_t count = ep0Pi.wCount.Val;
		if (count > EP0_BUFF_SIZE) {
			// 送信数がバッファより大きい
			count = EP0_BUFF_SIZE;
		}
		// 送信する転送数
		ep0Bi.CNT = count;
		// 残った転送数
		ep0Pi.wCount.Val -= count;
		// 転送が必要か？
		if (!ep0Pi.flag.NCPY) {
			// 転送先
			uint8_t* pDest = (uint8_t*)&DataPkt;
			while (count) {
				// ROMから転送
				if (ep0Pi.flag.ROM) *pDest++ = *ep0Pi.pSrc.ROM++;
				// RAMから転送
				else				*pDest++ = *ep0Pi.pSrc.RAM++;
				count--;
			}
		}
	}
	// 終了処理の移動 (2016/03/01)
	ep0Bi.ADR		= (uint8_t*)&DataPkt;
	ep0Bi.STAT.Val	= status;
	ep0Bi.STAT.UOWN	= 1;
}

/******************************************************************************
 ** スタンダートリクエスト
 ** リクエストのcase文は、1つの関数として分けることもできますが少しでもコードが
 ** 減るようにまとめてしています。
 ******************************************************************************/
void USBCheckStdRequest(void)
{
	// スタンダートディクエストか？
    if(SetupPkt.bmRequestType.RequestType != STANDARD) return;
	// リクエストの選択
	switch( SetupPkt.bRequest.Val ) {
		//** Get Status *******************************************************
        case GET_STATUS: {
			// 送信データデータ
			DataPkt.dat[0] = 0;
			DataPkt.dat[1] = 0;
			// 送信数
			ep0Pi.wCount.LB = 2;
			switch(SetupPkt.bmRequestType.Recipient) {
				case RCPT_DEV: {
					// Self-Powered Status [0] Bus-Powered [1] Self-Powered
					// RemoteWakeup        [0] Disabled    [1] Enabled
					if (self_power)			DataPkt.DEV.SPS = 1;
					if (usb_remote_wakeup)	DataPkt.DEV.RWP = 1;
					ep0Pi.flag.Val = PIPE_RAM | PIPE_NCPY | PIPE_BUSY;
		            break;
				}
				case RCPT_INTF: {
					// 処理無し(0x00, 0x00を返す)
					ep0Pi.flag.Val = PIPE_RAM | PIPE_NCPY | PIPE_BUSY;
					break;
				}
				case RCPT_EP: {
					if (SetupPkt.wIndex.EP.NUM < USB_USE_ENDPOINTS) {
						//  Halt Status [0] Not Halted [1] Halted
						uint8_t bd;
						// EPに対応するBDを算出
						if (SetupPkt.wIndex.EP.DIR)	bd = ep_BiNum(SetupPkt.wIndex.EP.NUM);
						else						bd = ep_BoNum(SetupPkt.wIndex.EP.NUM);
						// ストールしているか
						if ((BD[bd].STAT.UOWN == 1) && (BD[bd].STAT.BSTALL == 1))
							DataPkt.EP.HLT = 1;
						ep0Pi.flag.Val = PIPE_RAM | PIPE_NCPY | PIPE_BUSY;
					}
					break;
				}
		    }
            break;
		}
		//** Clear Feature ****************************************************
		case CLR_FEATURE: {
			switch(SetupPkt.bmRequestType.Recipient) {
				case RCPT_DEV: {
					if (SetupPkt.wValue.FETR.bFeature == DEVICE_REMOTE_WAKEUP) {
						usb_remote_wakeup = 0;
						ep0Pi.flag.Val = PIPE_ACK | PIPE_BUSY;
					}
					break;
				}
				case RCPT_EP: {
					if (SetupPkt.wValue.FETR.bFeature == ENDPOINT_HALT &&	
						SetupPkt.wIndex.EP.NUM > 0 &&
						SetupPkt.wIndex.EP.NUM < USB_USE_ENDPOINTS) {
						// EPに対応するBDを算出
						uint8_t bd;
						if (SetupPkt.wIndex.EP.DIR) {		// EPn IN
							bd = ep_BiNum(SetupPkt.wIndex.EP.NUM);
							BD[bd].STAT.Val =  _DAT0 | _DTSEN;
						}
						else {								// EPn Out
							bd = ep_BiNum(SetupPkt.wIndex.EP.NUM);
							BD[bd].STAT.Val = _USIE | _DAT0 | _DTSEN;
						}
						ep0Pi.flag.Val = PIPE_ACK | PIPE_BUSY;
					}
				}
				break;
		    }
			break;
		}
		//** Set Feature ******************************************************
		case SET_FEATURE: {
			switch(SetupPkt.bmRequestType.Recipient) {
				case RCPT_DEV: {
					if (SetupPkt.wValue.FETR.bFeature == DEVICE_REMOTE_WAKEUP) {
						usb_remote_wakeup = 1;
						ep0Pi.flag.Val = PIPE_ACK | PIPE_BUSY;
					}
					break;
				}
				case RCPT_EP: {
					if (SetupPkt.wValue.FETR.bFeature == ENDPOINT_HALT &&	
						SetupPkt.wIndex.EP.NUM > 0 &&
						SetupPkt.wIndex.EP.NUM < USB_USE_ENDPOINTS) {
						// EPに対応するBDを算出
						uint8_t bd;
						if (SetupPkt.wIndex.EP.DIR)	bd = ep_BiNum(SetupPkt.wIndex.EP.NUM);
						else 						bd = ep_BiNum(SetupPkt.wIndex.EP.NUM);
						BD[bd].STAT.Val = _USIE | _BSTALL;
						ep0Pi.flag.Val = PIPE_ACK | PIPE_BUSY;
					}
					break;
				}
			}
			break;
		}
		//** Set Address ******************************************************
		case SET_ADR: {
            USBDeviceState = ADR_PENDING_STATE;
			ep0Pi.flag.Val = PIPE_ACK | PIPE_BUSY;
            break;
		}
		//** Get Descriptor ***************************************************
		case GET_DSC: {
			// デバイスからホストに転送か?
			if (SetupPkt.bmRequestType.DataDir == DEV_TO_HOST) {
				// ディスクリプタ タイプの選択
				switch (SetupPkt.wValue.GDSR.bType) {
					//** Device Descriptor ************************************
					case DSC_DEV: {
						ep0Pi.pSrc.ROM = device_dsc;
						ep0Pi.wCount.LB = sizeof(device_dsc);
						ep0Pi.flag.Val = PIPE_ROM | PIPE_BUSY;
						break;
					}
					//** Configuration Descriptor *****************************
					case DSC_CFG: {
						if (SetupPkt.wValue.GDSR.bIndx == 0) {
							ep0Pi.pSrc.ROM = cfg_dsc01;
							ep0Pi.wCount.LB = sizeof(cfg_dsc01);
							ep0Pi.flag.Val = PIPE_ROM | PIPE_BUSY;
						}
						break;
					}
					case DSC_STR: {
					//** String Descriptor ************************************
						ep0Pi.pSrc.ROM = USB_SD_Ptr[ SetupPkt.wValue.GDSR.bIndx ];
						ep0Pi.wCount.LB = *ep0Pi.pSrc.ROM;
						ep0Pi.flag.Val = PIPE_ROM | PIPE_BUSY;
						break;
					}
					//** end of switch ****************************************
				}
			}
			break;
		}
		//** Set Descriptor ***************************************************
        //case SET_DSC: { break; } //(未サポート)
		//** Get Configuration ************************************************
		case GET_CFG: {
			ep0Pi.pSrc.RAM = (uint8_t*)&usb_active_cfg;
			ep0Pi.wCount.LB = 1;
			ep0Pi.flag.Val = PIPE_RAM | PIPE_BUSY;
			break;
		}
		//** Set Configuration ************************************************
		case SET_CFG: {
			// コンフィグ値を設定
			usb_active_cfg = SetupPkt.wValue.SCFG.bCfgVal;
			// 1つのコンフィグをサポート
			if (usb_active_cfg == 1) {	
				// EP1～EPnまで使用可にする。
				for (uint8_t i=1; i<USB_USE_ENDPOINTS; i++) {
					*(&UEP0 + i) = _OUT_IN | _HSHK;
					// EP番号からBDの番号を作成
					uint8_t bdo = ep_BoNum(i);
					uint8_t bdi = ep_BiNum(i);
					// 受信開始設定
					BD[bdo].CNT       = EPnOutBufSize[i];
					BD[bdo].ADR       = (uint8_t*)BDnBufAdr[bdo];
					BD[bdo].STAT.Val  = _DAT0 | _DTSEN;
					BD[bdo].STAT.UOWN = 1;
					// 送信設定
					BD[bdi].STAT.Val  = _DAT1 | _DTSEN;
				}
				// 通信開始
			    USBDeviceState = CONFIGURED_STATE;
			}
			else {
				// アドレスステートに戻る
				USBDeviceState = ADDRESS_STATE;
			}
			// 返答処理
			ep0Pi.flag.Val = PIPE_ACK | PIPE_BUSY;
            break;
		}
		//** Get Interface ****************************************************
        case GET_INTF: {
			ep0Pi.pSrc.RAM = (uint8_t*)usb_alt_intf +	  // 2016/01/20 &usb_alt_intfを修正 
										SetupPkt.wIndex.INTF.bIntfID;
            ep0Pi.wCount.LB = 1;
			ep0Pi.flag.Val = PIPE_RAM | PIPE_BUSY;
            break;
		}
		//** Set Interface ****************************************************
        case SET_INTF: {
            usb_alt_intf[SetupPkt.wIndex.INTF.bIntfID] = SetupPkt.wValue.INTF.bAltID;
			ep0Pi.flag.Val = PIPE_ACK | PIPE_BUSY;
            break;
		}
		//** Synch Frame ******************************************************
		//case SYNCH_FRAME: { break; } //(未サポート)
		//** end of switch ****************************************************
	}
}

/******************************************************************************
 ** HIDクラスリクエスト
 ** HID の構成(インターフェースが２つ等)によっては変更する必要があります。
 ******************************************************************************/
void USBCheckHIDRequest(void)
{
	// インターフェース？
	if(SetupPkt.bmRequestType.Recipient != RCPT_INTF) return;
	// インターフェースIDの確認
	if(SetupPkt.wIndex.INTF.bIntfID != HID_INTF_ID) return;
	// ディスクリプタの送信
	if(SetupPkt.bRequest.Val == GET_DSC) {
		switch(SetupPkt.wValue.GDSR.bType) {
			//** Get HID descriptor *******************************************
			case DSC_HID: {
				ep0Pi.pSrc.ROM = &cfg_dsc01[CFGDSC_SIZE	+ IFDSC_SIZE];
				ep0Pi.wCount.LB = HIDDSC_SIZE;
				ep0Pi.flag.Val = PIPE_RAM | PIPE_BUSY;
				break;
			}
			//** Get Report descriptor ****************************************
			case DSC_RPT:
				if(usb_active_cfg == 1) {
					ep0Pi.pSrc.ROM = hid_rpt01;
					ep0Pi.wCount.LB = sizeof(hid_rpt01); ;
					ep0Pi.flag.Val = PIPE_RAM | PIPE_BUSY;
				}
				break;
			//** Get Physical descriptor **************************************
			//case DSC_PHY: { break; } //(未サポート)
			//** end of switch ************************************************
		}
	}
	// クラスリクエスト
	if(SetupPkt.bmRequestType.RequestType == CLASS) {
		switch (SetupPkt.bRequest.Val) {
			//** GetReport ****************************************************
			case GET_REPORT: {
				break;
			}
			//** GetIdle ******************************************************
			case GET_IDLE: {
				ep0Pi.pSrc.RAM = (uint8_t*)&usb_idle_rate;
				ep0Pi.wCount.LB = 1;
				ep0Pi.flag.Val = PIPE_RAM | PIPE_BUSY;
				break;
			}
			//** GetProtocol **************************************************
			case GET_PROTOCOL: {
				ep0Pi.pSrc.RAM = (uint8_t*)&usb_active_protocol;
				ep0Pi.wCount.LB = 1;
				ep0Pi.flag.Val = PIPE_RAM | PIPE_BUSY;
				break;
			}
			//** SetReport ****************************************************
			//case SET_REPORT: { break; } //(未サポート)
			//** SetIdle ******************************************************
			case SET_IDLE: {
				usb_idle_rate = SetupPkt.wValue.SIDL.bDuration;
				ep0Pi.flag.Val = PIPE_ACK | PIPE_BUSY;
				break;
			}
			//** SetProtocol **************************************************
			case SET_PROTOCOL: {
				usb_active_protocol = SetupPkt.wValue.SPTL.bProtocol;
				ep0Pi.flag.Val = PIPE_ACK | PIPE_BUSY;
				break;
			}
			//** end of switch ************************************************
		}
	}
}

/** EOF usbhiddev.c ***********************************************************/
