/****************************************************************************** 
 ** AE-PIC18F14K50 Monitor Program					Create Date	2015/11/20
 **				for PIC18F14K50						Last Update	2016/03/12
 ** file name :  ae-18f14k50.c								Saka Softwares
 ** language  :  XC8 (v1.35)
 ******************************************************************************
 ** 新型2KB版 HID bootloadr 及び usbhiddev.c を使っています。
 ** 動作確認は、AE-Controll Version 1.5で行っています。
 ** このプログラムの目的は、C18言語からXC8言語に移行するために作りました。
 ** のでC18でのコンパイルは出来ません。
 ** usbhiddev.c に対応するI/O制御プログラムです。
 ** ボードは秋月電子通商の AE-PIC18F14K50(K-05499) に対応します。
 ******************************************************************************/
/** Include Files *************************************************************/
#include <xc.h>
#include <stdint.h>
#include <stdbool.h>

#include "usbhiddev.h"

/** Version info. *************************************************************/
const uint16_t VersionWord @ 0x0816 = 0x0220;	// BCD表現のバージョン No.
#define	HARDWEAR_VERSION	"AE-PIC18F14K50 Monitor Ver2.2"
/******************************************************************************
 **	2.2		2016/03/12	usbhiddevの不具合修正
 **	2.0		2015/11/20	XC8対応バージョン
 **						以降 旧バージョン
 **	1.7		2013/09/29	割り込み禁止/受信バグ修正/連続入力追加
 **	1.0		2013/06/27	公開バージョン
 ******************************************************************************/

/** Configuration *************************************************************/
// HIDブートローダーと同一コンフィグレーションにしてください。
// pic18f14k50_xc8_bootloader.hexに対応
#pragma config  CPUDIV	= NOCLKDIV	// CPU システムクロック選択ビット
#pragma config  USBDIV	= OFF		// USB クロック選択ビット
#pragma config  FOSC	= HS		// オシレータ選択ビット
#pragma config  PLLEN	= ON		// 4 X PLL イネーブルビット
#pragma config  PCLKEN	= OFF		// プライマリ クロック イネーブルビット
#pragma config  FCMEN	= OFF		// フェイルセーフ クロック モニタ イネーブルビット
#pragma config  IESO	= OFF		// 内部/ 外部オシレータ切り換えビット
#pragma config  PWRTEN	= OFF		// パワーアップ タイマ イネーブルビット (OFFで有効)
#pragma config  BOREN	= ON		// ブラウンアウト リセット イネーブルビット
#pragma config  BORV	= 30		// ブラウンアウト リセット電圧ビット
#pragma config  WDTEN	= OFF		// ウォッチドッグ タイマ イネーブルビット
#pragma config  WDTPS	= 32768		// ウォッチドッグ タイマ ポストスケーラ選択ビット
#pragma config  HFOFST	= OFF		// HFINTOSC 高速起動ビット
#pragma config  MCLRE	= OFF		// MCLR ピン イネーブルビット
#pragma config  STVREN	= ON		// スタックフル/ アンダーフロー リセット イネーブルビット
#pragma config  LVP		= OFF		// 単電源ICSP イネーブルビット
#pragma config  BBSIZ	= ON		// ブートブロック サイズ選択ビット(2kW boot block size)
#pragma config  XINST	= OFF		// 拡張命令セットイネーブルビット (ENHCPU)
#pragma config  CP0		= OFF		// コード保護ビット
#pragma config  CP1		= OFF		// コード保護ビット
#pragma config  CPB		= OFF		// ブートブロック コード保護ビット
#pragma config  CPD		= OFF		// EEPROM コード保護ビット
#pragma config  WRT0	= OFF		// 書き込み保護ビット
#pragma config  WRT1	= OFF		// 書き込み保護ビット
#pragma config  WRTC	= OFF		// コンフィグレーション レジスタ書き込み保護ビット
#pragma config  WRTB	= OFF		// ブートブロック書き込み保護ビット
#pragma config  WRTD	= OFF		// EEPROM 書き込み保護ビット
#pragma config  EBTR0	= OFF		// テーブル読み出し保護ビット
#pragma config  EBTR1	= OFF		// テーブル読み出し保護ビット
#pragma config  EBTRB	= OFF		// ブートブロック テーブル読み出し保護ビット

/** Hardware Settings *********************************************************/
// PortB, PortCは、プログラム内で直接操作します。
// SWITCH
#define sw2			PORTAbits.RA3

/** Variable ******************************************************************/
#define DATA_SIZE	64			// EP1Outと同じか大きくしてください。
uint8_t rcv_dat[DATA_SIZE];		// 受信データ
uint8_t snd_dat[DATA_SIZE];		// 送信データ

// 割り込み用送信データ		0123456789012345678901234567
uint8_t inp_data[] =	 { "inp a 0x0000, b 0x00, c 0x00" };

union {
	uint8_t val;				//フラグ値
	struct {
		unsigned SEND:1;		//送信フラグ
		unsigned ATPT:1;		//ポート自動送信フラグ
		unsigned ATAD:1;		//ADC自動送信フラグ
		unsigned :6;
	};
} FLAG;							// 各種フラグ

/** Internal Function *********************************************************/
void main(void);
void initPorts (void);
void initIrq (void);
void IrqTimer2 (void);
void interrupt high_priority IsrHigh(void);
void interrupt low_priority IsrLow(void);

void ProcessIO(void);
void bootloader(uint8_t* pRcv);

uint8_t* ByteToHex (uint8_t dat, uint8_t *p);
uint8_t* WordToHex (uint16_t dat, uint8_t *p);
uint8_t _NibbleToByte (uint8_t dat);
uint8_t HexToByte (uint8_t *p);
uint16_t HexToWord(uint8_t *p);

void Version (void);
void AutoSend(void);
void AutoOFF (void);
void AutoSetPort (void);
void AutoSetADC (uint8_t *pRcv);
void SetTris (uint8_t *pRcv);
void SetPorts (uint8_t *pRcv);
void GetPorts (void);
void SetADCs (uint8_t *pRcv);
void GetADC (uint8_t *pRcv);				

void MainUsbStatus(void);

/******************************************************************************
 ** メイン処理部
 ** 引  数	: 無し
 ** 戻り値	: 無し
 ******************************************************************************/
void main(void)
{
	ClrWdt();
	INTCON  = 0x00;					// 割り込み禁止
	STKPTR  = 0x00;					// スタックポインタ設定
	WDTCON  = 0x01;					// ウォッチドックタイマ起動
	FLAG.val = 0x00;				// フラグクリア
	initPorts();					// ボートの初期化
	USBDeviceInit();				// 割り込み対応USBの初期化
	initIrq();						// 割り込みの初期化
	while (1) {
		ClrWdt();
		bootloader(rcv_dat);		// ブートローダー移行処理を行います。
		USBDeviceTasks();			// USBのバス接続切断処理を行います。
		MainUsbStatus();			// USBフラグを見てLEDの点灯処理を行います。
		ProcessIO();				// 通信処理
    }    
}

/******************************************************************************
 ** ポートの初期化
 ** 引  数	: 無し
 ** 戻り値	: 無し
 ******************************************************************************/
void initPorts (void)
{
	PORTB = 0x00;
	LATB  = 0x00;
	TRISB = 0xF0;
	WPUB  = 0x00;
	PORTC = 0x00;
	TRISC = 0xFF;
	LATC  = 0x00;
	ANSEL = 0x00;
}

/******************************************************************************
 ** 割り込みの初期化
 ** 引  数	: 無し
 ** 戻り値	: 無し
*******************************************************************************/
void initIrq (void)
{
	// USB割り込み
	IPR2bits.USBIP	= 0;		// USB 割り込み優先度:Low
	// Timer2 init				// Fosc 48MHz
	PR2 = 149;					// (149 + 1)* (1/(Fosc/4/16)) = 200uS
	T2CONbits.T2OUTPS = 9;		// Select Postscale 1:10 (2mS Interrupt)
	T2CONbits.T2CKPS  = 2;		// Select Prescaler 1:16 (Fosc/4/16)
	IPR1bits.TMR2IP   = 1;		// TIMER2 Interrupt High Priority
	PIR1bits.TMR2IF   = 0;		// TIMER2 Interrupt flag clear
	PIE1bits.TMR2IE   = 1;		// TIMER2 Interrupt enable
	// 割り込み
	RCONbits.IPEN     = 1;		// 優先度付き割り込み
	INTCONbits.GIEL   = 1;		// 低割り込み許可
	INTCONbits.GIEH   = 1;		// 高割り込み許可
}

/******************************************************************************
 ** 高レベル割り込み処理部
 ** 引  数	: 無し
 ** 戻り値	: 無し
 ** タイマ2の自動送信処理を行います。 
 ******************************************************************************/
void interrupt high_priority IsrHigh(void)
{
	// タイマ2割り込み処理
	IrqTimer2();
}

/******************************************************************************
 ** 低レベル割り込み処理部
 ** 引  数	: 無し
 ** 戻り値	: 無し
 ** USBのプラグ&プレイ関連の処理を行っています。必ず必要です。 
 ******************************************************************************/
void interrupt low_priority IsrLow(void)
{
	// USB割り込み処理
	USBInterruptTasks();
}

/******************************************************************************
 ** プロセスＩＯ処理部
 ** 引  数	: 無し
 ** 戻り値	: 無し
 ** USB関連の処理を一括で行っています。
 ** HID受信した場合フラグを立てて各処理を行い、HID送信が可能になるまで待ち
 ** 結果を返す仕様にしています。
 ** 1回の受信または送信は64byteです。HIDのリポートディスクリプタで設定しています。
 ** 送信処理(HidTxRamReport)は、前の送信が終わるまで待ちます。
 ** 複数送る場合ウォッチドックのタイムアウトに気を付けてください。
 ******************************************************************************/
void ProcessIO(void)
{
	// 受信処理
	if (HidRxReport(rcv_dat, DATA_SIZE) < RCV_RECEING) {
		switch (rcv_dat[0]) {
			case 'v' : Version();				break;	// ハードウェアバージョンの送信
			case 'i' : initPorts();				break;	// ポートの初期化
			case 'g' : GetPorts();				break;	// ポート情報の取得
			case 's' : SetTris(rcv_dat);		break;	// ポート入出力方向設定
			case 'p' : SetPorts(rcv_dat);		break;	// ポート出力設定
			case 'f' : SetADCs(rcv_dat);		break;	// ADC設定
			case 'a' : GetADC(rcv_dat);			break;	// ADC入力
			case 'P' : AutoSetPort();			break;	// 連続ポート情報の取得
			case 'A' : AutoSetADC(rcv_dat);		break;	// 連続ADC情報の取得
			case 'T' : AutoOFF();				break;	// 自動送信終了処理
			default	 : break;
		}
	}
	AutoSend ();										// 自動送信処理
}

/******************************************************************************
 ** ブートローダー処理部
 ** 引  数	: 受信データバッファポインタ
 ** 戻り値	: 無し
 ** sw2または受信コマンドによりHIDブートローダーへ移行します。
 ** PIC 新型2KB版 HID Bootloaderに対応しています。
 ******************************************************************************/
void bootloader(uint8_t* pRcv)
{
	if (!sw2 || (pRcv != 0 && *(pRcv+0) == 'B'
		&& *(pRcv+1) == 'o' && *(pRcv+2) == 'o' && *(pRcv+3) == 't')) {
		INTCONbits.GIEL = 0;				// 低割り込み禁止
		INTCONbits.GIEH = 0;				// 高割り込み禁止
		LATA  = 0;							// ポートAクリア
		LATB  = 0;							// ポートBクリア
		LATC  = 0;							// ポートcクリア
		TRISA = 0;							// ポートA初期化
		TRISB = 0;							// ポートB初期化
		TRISC = 0;							// ポートC初期化
		uint16_t cnt = 0xFFFF;
		while (cnt--) { 
			ClrWdt();
			UCON = 0;						// ウェイト付きUSB停止
		}
		asm("goto 0x001C");					// bootloader Entryへ
	}
}

/******************************************************************************
 * 文字列変換 (16進数の文字列変換します。)
 ** 引  数	: 変換前の 数値 または 文字列
 ** 戻り値	: 変換後の 文字列 または 数値
 ******************************************************************************/
const static uint8_t HEX16[] = { "0123456789ABCDEF" };
// 数値(1byte)から文字列(2byte)へ
uint8_t* ByteToHex (uint8_t dat, uint8_t *p)
{
	*(p++) = HEX16[ (dat>>4)&0x0F ];
	*(p++) = HEX16[ (dat   )&0x0F ];
	return p;
}
// 数値(1word)から文字列(4byte)へ
uint8_t* WordToHex (uint16_t dat, uint8_t *p)
{
	*(p++) = HEX16[ (dat>>12)&0x0F ];
	*(p++) = HEX16[ (dat>> 8)&0x0F ];
	*(p++) = HEX16[ (dat>> 4)&0x0F ];
	*(p++) = HEX16[ (dat    )&0x0F ];
	return p;
}
// 1文字から1byteへ(内部処理用)
uint8_t _NibbleToByte (uint8_t dat)
{
	if ('0'<=dat && dat<='9')		return (dat-'0');
	else if ('a'<=dat && dat<='f')	return (dat-'a'+10);
	else if ('A'<=dat && dat<='F')	return (dat-'A'+10);
	return 0;
}
// 文字列(2yte)から数値(1byte)へ
uint8_t HexToByte (uint8_t *p)
{
	return (_NibbleToByte(*(p+0))<<4) + _NibbleToByte(*(p+1));
}
// 文字列(4yte)から数値(1word)へ
uint16_t HexToWord(uint8_t *p)
{
	return (HexToByte(p+0)<<8) + HexToByte(p+2);
}

/********************************************************************
 ** バージョンの送信
 ** 引  数	: 無し
 ** 戻り値	: 無し
 *******************************************************************/
void Version (void)
{
	HidTxRomReport(HARDWEAR_VERSION, sizeof(HARDWEAR_VERSION));
}

/********************************************************************
 *** 自動送信
 ** 引  数	: 無し
 ** 戻り値	: 無し
 *******************************************************************/
void AutoSend(void)
{
	if (FLAG.SEND) {
		FLAG.SEND = 0;
		// 送信
		HidTxRamReport(inp_data, sizeof(inp_data));
	}
}

/********************************************************************
 ** 自動送信の終了
 ** 引  数	: 無し
 ** 戻り値	: 無し
 *******************************************************************/
void AutoOFF (void)
{
	FLAG.ATAD = 0;				// ADCデータ送信フラグ OFF
	FLAG.ATPT = 0;				// Prot入力データ送信フラグ OFF
	T2CONbits.TMR2ON  = 0;		// Timr2 OFF
}

/********************************************************************
 ** 自動Prot入力データ送信
 ** 引  数	: 無し
 ** 戻り値	: 無し
 *******************************************************************/
void AutoSetPort (void)
{
	FLAG.ATPT = 1;				// Prot入力データ送信フラグ ON
	T2CONbits.TMR2ON  = 1;		// Timr2 ON
}

/********************************************************************
 ** 自動ADC入力データ送信
 ** 引  数	: 受信データ
 ** 戻り値	: 無し
 *******************************************************************/
void AutoSetADC (uint8_t *pRcv)	// "A 0x00"
{
	uint8_t ch = HexToByte(pRcv+4);
	if (4<=ch && ch<=11) {
		ADCON0bits.CHS = ch << 2;
		FLAG.ATAD = 1;				// ADCデータ送信フラグ ON
		T2CONbits.TMR2ON  = 1;		// Timr2 ON
	}
}

/********************************************************************
 ** タイマ2の割り込み処理
 ** 引  数	: 無し
 ** 戻り値	: 無し
 *******************************************************************/
void IrqTimer2 (void)
{
	if (PIR1bits.TMR2IF) {
		PIR1bits.TMR2IF = 0;				// Timer2割り込みクリア
		if (!FLAG.SEND) {
			if (FLAG.ATAD) {
				ADCON0bits.ADON = 1;		// ADC ON
				ADCON0bits.GO   = 1;
				while (ADCON0bits.DONE);
				ADCON0bits.ADON = 1;		// ADC OFF
				WordToHex(ADRES, inp_data+8);
				FLAG.SEND = 1;				// 送信フラグ
			}
			else {
				WordToHex(0, inp_data+8);
			}
			if (FLAG.ATPT) {
				ByteToHex(PORTB, inp_data+18);
				ByteToHex(PORTC, inp_data+26);
				FLAG.SEND = 1;				// 送信フラグ
			}
			else {
				ByteToHex(0, inp_data+18);
				ByteToHex(0, inp_data+26);
			}
		}
	}
}

/********************************************************************
 ** 固定データを送信バッファに転送
 ** 引  数	: 送信する文字列
 ** 戻り値	: 無し
 *******************************************************************/
uint8_t Setsnd_dat(const uint8_t *pDat)
{
	uint8_t cnt;
	for (cnt=0; cnt<DATA_SIZE; cnt++) {
		snd_dat[cnt] = *pDat++;
		if (snd_dat[cnt]=='\0') break;
	}
	return cnt;	
}

/********************************************************************
 ** ポートの設定
 ** 引  数	: 受信データ
 ** 戻り値	: 無し
 *******************************************************************/
void SetTris (uint8_t *pRcv)	// 01234567890123456789
{								//"s b 0x00 0x00, c 0x00"
	if (*(pRcv+2)=='b') {	// ポートB
		TRISB = HexToByte((char*)pRcv+6);
		WPUB  = HexToByte((char*)pRcv+11);
	}
	if (*(pRcv+15)=='c') {	// ポートC
		TRISC = HexToByte((char*)pRcv+19);
	}
}

/********************************************************************
 ** ポートの出力
 ** 引  数	: 受信データ
 ** 戻り値	: 無し
 *******************************************************************/
void SetPorts (uint8_t *pRcv)	// 01234567890123456789
{								//"p l 0x00, l 0x00"
	// ポートB
	if (*(pRcv+2)=='l')		LATB  = HexToByte(pRcv+6);
	if (*(pRcv+2)=='p')		PORTB = HexToByte(pRcv+6);
	// ポートC
	if (*(pRcv+10)=='l')	LATC  = HexToByte(pRcv+14);
	if (*(pRcv+10)=='p')	PORTC = HexToByte(pRcv+14);
}

/********************************************************************
 ** ポート情報の読み出し
 ** 引  数	: 無し
 ** 戻り値	: 無し
 *******************************************************************/
void GetPorts (void)		 //  0         1         2         3         4
{							 //  01234567890123456789012345678901234567890
	uint8_t ports =	Setsnd_dat( "b 0x00 0x00 0x00 0x00, c 0x00 0x00 0x00" );
	// ポートB
	ByteToHex(PORTB, snd_dat+4);
	ByteToHex(TRISB, snd_dat+9);
	ByteToHex(LATB,  snd_dat+14);
	ByteToHex(WPUB,  snd_dat+19);
	// ポートC	
	ByteToHex(PORTC, snd_dat+27);
	ByteToHex(TRISC, snd_dat+32);
	ByteToHex(LATC,  snd_dat+37);
	// 送信
	HidTxRamReport(snd_dat, ports);
}

/********************************************************************
 ** ADC設定
 ** 引  数	: 受信データ
 ** 戻り値	: 無し
 *******************************************************************/
void SetADCs (uint8_t *pRcv)// 01234567890123456
{							//"f a 0x0000"
	if (*(pRcv+2)=='a') {
		ANSEL = HexToWord(pRcv+6);
		if (ANSEL==0) {
			FLAG.ATAD			= 0;	// ADCデータ送信フラグ OFF
			ADCON0bits.ADON		= 0;	// ADC OFF
			REFCON0bits.FVR1EN	= 0;	// FVR OFF
		}
		else {
			TRISB |= (ANSEL>>6)&0x30;
			TRISC |= (ANSEL>>2)&0xC0 + (ANSEL>>4)&0x0F;
			ADCON1 = 0b00001000;		// selected FVR/VSS
			ADCON2 = 0b10110110;		// right/16TAD/Fosc/64 (TAD = 1.33uS)
			REFCON0 = 0b10110000;		// FVR ON / 4.096V
		}
	}
}

/********************************************************************
 ** ADCデータの読み込み
 ** 引  数	: 受信データ
 ** 戻り値	: 無し
 *******************************************************************/
void GetADC (uint8_t *pRcv)	// "a 0x00"					
{							//  0123456789012345678
	uint8_t adc =  Setsnd_dat( "adc 0x0000 ch 0x00" );
	// 受信データからチャンネルの取得
	uint8_t ch = HexToByte(pRcv + 4);
	// 送信データにチャンネルの書き込み
	ByteToHex(ch, snd_dat + 16);
	// ADCの情報取得
	if (4<=ch && ch<=11) {
		ADCON0bits.CHS = ch << 2;
		ADCON0bits.ADON = 1;		// ADC ON
		ADCON0bits.GO   = 1;
		while (ADCON0bits.DONE);
		ADCON0bits.ADON = 1;		// ADC OFF
		// 送信データにADC値の書き込み
		WordToHex(ADRES, snd_dat + 6);
	}
	// 送信
	HidTxRamReport(snd_dat, adc);
}

/******************************************************************************
 ** USBステータスによる各処理部
 ** 引  数	: 無し
 ** 戻り値	: 無し
 ** サスペンド処理
 ** サスペンド時にスリープします。
 ** その前後にユーザー処理を入れます。
 ******************************************************************************/
void MainUsbStatus(void)
{
	//** サスペンド処理 ********************************************************
	//** サスペンド中 USBのバスパワーの供給は、500μAと規定されています。
	//** バスパワーで動いている機器は、LEDの点灯はもちろんCPUの通常稼働も
	//** できないでしょう。(守っていない機器も多いみたいだけど。)
	//** セルフパワーならあまり関係ないかもしれませんが。
	if (USBGetDeviceState() == CONFIGURED_STATE && USBIsSuspended() ) {
		uint8_t blb, blc, bfg;
		// メインのサスペンド移行処理部
		blb = LATB;						// PortB点灯状態のバックアップ
		blc = LATC;						// PortC点灯状態のバックアップ
		bfg = FLAG.val;					// フラグのバックアップ
		LATB		= 0;				// 消灯
		LATC		= 0;				// 消灯
		FLAG.val	= 0;				// 自動通信停止
		T2CONbits.TMR2ON  = 0;			// Timr2 OFF
		// サスペンド中 USBのアクティブを待って何もせずスリープする
		while (!USBIsActive() && !RCONbits.TO) {
			Sleep();
		}
		// メインのウェイクアップ処理部
		LATB		= blb;				// PortB点灯状態の復帰
		LATC		= blc;				// PortB点灯状態の復帰
		FLAG.val	= bfg;				// フラグの復帰
		if (FLAG.ATAD || FLAG.ATPT) {
			T2CONbits.TMR2ON  = 0;		// Timr2 ON
		}
	}
}

/** EOF ae-18f14k50.c *********************************************************/
